/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-05-25
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <GeopsyCore.h>
#include <QGpGuiTools.h>

#include "SortKeysModel.h"

/*!
  \class SortKeysModel SortKeysModel.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

const QString SortKeysModel::ascending(tr("ascending"));
const QString SortKeysModel::descending(tr("decending"));

/*!
  Description of constructor still missing
*/
SortKeysModel::SortKeysModel(QObject * parent)
  : QAbstractTableModel(parent)
{
  TRACE;
}

/*!
  Description of destructor still missing
*/
SortKeysModel::~SortKeysModel()
{
  TRACE;
}

int SortKeysModel::rowCount(const QModelIndex&) const
{
  TRACE;
  return SortKey::count();
}

int SortKeysModel::columnCount(const QModelIndex&) const
{
  TRACE;
  return 3;
}

QVariant SortKeysModel::data(const QModelIndex& index, int role) const
{
  TRACE;
  const SortKey& key=SortKey::at(index.row());
  switch (role) {
  case Qt::DisplayRole:
    switch (index.column()) {
    case 0: return MetaDataFactory::instance()->baseName(key);
    case 1: return key.index();
    case 2: return key.reversed() ? descending : ascending;
    default: return QVariant();
    }
  default:
    return QVariant();
  }
}

bool SortKeysModel::setData(const QModelIndex& index, const QVariant & value,
                                     int role)
{
  TRACE;
  SortKey& key=SortKey::at(index.row());
  switch (role) {
  case Qt::EditRole:
    switch (index.column()) {
    case 0: {
        MetaDataIndex index=MetaDataFactory::instance()->index(value.toString());
        if(index.isValid()) {
          key=index;
          return true;
        } else {
          return false;
        }
      }
    case 1:
      key.setIndex(value.toString());
      return true;
    case 2:
      key.setReversed(value.toString()==descending);
      return true;
    default:
      return false;
    }
  default:
    return false;
  }
}

QVariant SortKeysModel::headerData(int section, Qt::Orientation orientation,
                                            int role) const
{
  TRACE;
  switch (role) {
  case Qt::DisplayRole:
    if(orientation==Qt::Horizontal) {
      switch (section) {
      case 0: return tr("Key");
      case 1: return tr("Index");
      case 2: return tr("Order");
      default: return QVariant();
      }
    } else {
      return section+1;
    }
  default:
    return QVariant();
  }
}

Qt::ItemFlags SortKeysModel::flags (const QModelIndex& index) const
{
  const SortKey& key=SortKey::at(index.row());
  if(index.column()!=1 || MetaDataFactory::instance()->isArray(key)) {
    return Qt::ItemIsEnabled | Qt::ItemIsEditable | Qt::ItemIsSelectable;
  } else {
     return Qt::ItemIsSelectable;
  }
}


void SortKeysModel::add()
{
  TRACE;
  int n=SortKey::count();
  beginInsertRows(QModelIndex(), n, n);
  if(n>0) {
    SortKey::add(SortKey::at(n-1));
  } else {
    SortKey::add(SortKey(MetaDataFactory::Name));
  }
  endInsertRows();
}

void SortKeysModel::remove(QModelIndexList l)
{
  TRACE;
  QList<int> rows=sortedRows(l);
  for(int i=rows.count()-1; i>=0; i--) {
    int row=rows.at(i);
    beginRemoveRows(QModelIndex(), row, row);
    SortKey::remove(row);
    endRemoveRows();
  }
}

QModelIndexList SortKeysModel::moveUp(QModelIndexList l)
{
  TRACE;
  QList<int> rows=sortedRows(l);
  if(rows.isEmpty() || rows.first()==0) { // already on top
    return l;
  }
  l.clear();
  for(int i=0; i<rows.count(); i++) {
    int row=rows.at(i);
    beginMoveRows(QModelIndex(), row, row, QModelIndex(), row-1);
    SortKey::moveUp(row);
    endMoveRows();
    l.append(index(row-1, 0));
  }
  return l;
}

QModelIndexList SortKeysModel::moveDown(QModelIndexList l)
{
  TRACE;
  QList<int> rows=sortedRows(l);
  if(rows.isEmpty() || rows.last()==SortKey::count()-1) { // already at bottom
    return l;
  }
  l.clear();
  for(int i=rows.count()-1; i>=0; i--) {
    int row=rows.at(i);
    beginMoveRows(QModelIndex(), row, row, QModelIndex(), row+2);
    SortKey::moveDown(row);
    endMoveRows();
    l.append(index(row+1, 0));
  }
  return l;
}

void SortKeysModel::load()
{
  TRACE;
  QSettings& reg=CoreApplication::instance()->settings();
  QStringList keysStr=reg.value("Sort Keys").toStringList();
  beginResetModel();
  SortKey::clear();
  for(QStringList::Iterator it=keysStr.begin(); it!=keysStr.end(); ++it) {
    QString key=(*it).mid(1);
    if((*it)[0]=='-') {
      SortKey::add(SortKey(MetaDataFactory::instance()->index(key), true));
    } else {
      SortKey::add(SortKey(MetaDataFactory::instance()->index(key), false));
    }
  }
  endResetModel();
}

void SortKeysModel::save()
{
  TRACE;
  QStringList keysStr;
  int n=SortKey::count();
  for(int i=0; i<n; i++) {
    const SortKey& key=SortKey::at(i);
    if(key.reversed()) {
      keysStr.append("-"+MetaDataFactory::instance()->name(key));
    } else {
      keysStr.append("+"+MetaDataFactory::instance()->name(key));
    }
  }
  QSettings& reg=CoreApplication::instance()->settings();
  reg.setValue("Sort Keys", keysStr);
}
