/***************************************************************************
**
**  This file is part of geopsy.
**
**  geopsy is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  geopsy is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-08-30
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "Engine.h"
#include "SeismicEventView.h"
#include "SeismicEventDelegate.h"
#include "PickEvents.h"

/*!
  \class SeismicEventView SeismicEventView.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
SeismicEventView::SeismicEventView(QWidget * parent)
  : TableView(parent)
{
  TRACE;
  _addAction=nullptr;
  _pickAction=nullptr;
  _removeAction=nullptr;
  _importAction=nullptr;
  _exportAction=nullptr;

  _menu=new QMenu(this);
  setContextMenuPolicy(Qt::DefaultContextMenu);

  setItemDelegate(new SeismicEventDelegate(this));

  SeismicEventItem * model=new SeismicEventItem(this);
  model->setDatabase(Engine::instance()->database(this));
  setModel(model);
  connect(model, SIGNAL(eventsChanged()), this, SIGNAL(eventsChanged()));

  setSortingEnabled(true);
}

/*!
  Description of destructor still missing
*/
SeismicEventView::~SeismicEventView()
{
  TRACE;
  Settings::setColumnWidth(this, "SeismicEventView");
}

inline SeismicEventItem * SeismicEventView::model() const
{
  return static_cast<SeismicEventItem *>(TableView::model());
}

bool SeismicEventView::event(QEvent *event)
{
  TRACE;
  if(event->type()==QEvent::Polish) {
    resizeColumnsToContents();
    // Not OK in the constructor because object name not yet set
    Settings::columnWidth(this, "SeismicEventView");
  }
  return TableView::event(event);
}

void SeismicEventView::changeAllRows(const QModelIndex& index, StringDiff string)
{
  // Selecting all with the same time and position makes crashing
  if(index.column()>=5) {
    TableView::changeAllRows(index, string);
  }
}

void SeismicEventView::addActions()
{
  TRACE;
  _addAction=new QAction(tr("Add"), this);
  _addAction->setStatusTip(tr("Adds a new empty event in the table"));
  connect(_addAction, SIGNAL(triggered()), this, SLOT(addEvent()));

  _pickAction=new QAction(tr("Pick"), this);
  _pickAction->setStatusTip(tr("Automatically picks events on a signal assumed to be located at the source."));
  connect(_pickAction, SIGNAL(triggered()), this, SLOT(pickEvents()));

  _removeAction=new QAction(tr("Remove"), this);
  _removeAction->setStatusTip(tr("Removes the selected events."));
  connect(_removeAction, SIGNAL(triggered()), this, SLOT(removeEvents()));

  _importAction=new QAction(tr("Import"), this);
  _importAction->setStatusTip(tr("Imports events from a file."));
  connect(_importAction, SIGNAL(triggered()), this, SLOT(importEvents()));

  _exportAction=new QAction(tr("Export"), this);
  _exportAction->setStatusTip(tr("Exports selected events to a file."));
  connect(_exportAction, SIGNAL(triggered()), this, SLOT(exportEvents()));
}

QList<SeismicEvent *> SeismicEventView::getSelection() const
{
  TRACE;
  QList<SeismicEvent *> sel;
  int n=model()->rowCount(QModelIndex());
  for(int i=0; i<n; i++) {
    QModelIndex m=model()->index(i, 0, QModelIndex());
    if(selectionModel()->isSelected(m)) {
      sel.append(model()->at(i));
    }
  }
  return sel;
}

void SeismicEventView::contextMenuEvent(QContextMenuEvent * e)
{
  TRACE;
  _menu->clear();
  QList<SeismicEvent *> sel=getSelection();
  bool hasLockedItems=false;
  for(QList<SeismicEvent *>::iterator it=sel.begin(); it!=sel.end(); it++) {
    if((*it)->isLocked()) {
      hasLockedItems=true;
      break;
    }
  }

  _menu->addAction(_addAction);
  _menu->addAction(_pickAction);
  if(!sel.isEmpty() && !hasLockedItems) {
    _menu->addAction(_removeAction);
  }
  _menu->addSeparator();
  _menu->addAction(_importAction);
  if(!sel.isEmpty()) {
    _menu->addAction(_exportAction);
  }
  _menu->popup(viewport()->mapToGlobal(e->pos()));
}

void SeismicEventView::addEvent()
{
  TRACE;
  int col=0;
  int i=currentIndex().row();
  SeismicEvent e;
  if(i>=0) {
    e=*model()->at(i);
    DateTime t=e.time();
    t.addMinutes(1);
    e.setTime(t);
    Point p=e.position();
    p.setX(p.x()+1.0);
    e.setPosition(p);
    col=currentIndex().column();
  }
  Engine::instance()->beginSignalChange(model()->dataBase());
  model()->dataBase()->seismicEvents()->add(e);
  i=model()->rowCount(QModelIndex())-1;
  setCurrentIndex(model()->index(i, col));
  Engine::instance()->endSignalChange(model()->dataBase());
}

void SeismicEventView::pickEvents()
{
  TRACE;
  SignalDatabase * db=model()->dataBase();
  PickEvents * d=new PickEvents(this);
  d->setSubPool(WindowEnvironment::instance()->window(db)->currentSubPoolWin());
  Settings::getWidget(d);
  d->updateAllFields();
  Engine::instance()->beginSignalChange(model()->dataBase());
  if(d->exec()==QDialog::Accepted) {
    Settings::setWidget(d);
    d->pickEvents();
  } else {
    d->removeEvents();
  }
  delete d;
  Engine::instance()->endSignalChange(model()->dataBase());
}

void SeismicEventView::removeEvents()
{
  TRACE;
  QList<SeismicEvent *> toRemove=getSelection();
  if(!toRemove.isEmpty()) {
    selectionModel()->clear();
    SeismicEventTable * events=model()->dataBase()->seismicEvents();
    Engine::instance()->beginSignalChange(model()->dataBase());
    for(int i=toRemove.count()-1; i>=0; i--) {
      events->remove(*toRemove.at(i));
    }
    Engine::instance()->endSignalChange(model()->dataBase());
  }
}

void SeismicEventView::importEvents()
{
  TRACE;
  QString fileName=Message::getOpenFileName(tr("Import seimsic events"),
                                            tr("Seismic events (*.evt)"));
  if(!fileName.isEmpty()) {
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      Engine::instance()->beginSignalChange(model()->dataBase());
      QTextStream s(&f);
      LineParser lp;
      bool ok=true;
      DateTime time;
      int lineNumber=0;
      SeismicEventTable * events=model()->dataBase()->seismicEvents();
      SeismicEvent event;
      while(!s.atEnd() && ok) {
        QString buf=s.readLine();
        lineNumber++;
        if(!buf.isEmpty() && buf[0]!='#') {
          lp.setString(buf);

          QString timeS=lp.toString(0, ok);
          if(!ok || !time.fromString(timeS, DateTime::defaultFormat)) {
            App::log(tr("Error reading time specification at line %1\n").arg(lineNumber));
            ok=false;
          }
          event.setTime(time);

          QString utmS=lp.toString(1, ok);
          if(!ok) {
            App::log(tr("Error reading utm zone at line %1\n").arg(lineNumber));
          }
          UtmZone utm;
          utm.fromString(utmS);
          if(!utmS.isEmpty() && !utm.isValid()) {
            App::log(tr("Error parsing utm zone at line %1: '%2'\n").arg(lineNumber).arg(utmS));
            ok=false;
          }
          event.setUtmZone(utm);

          event.setPosition(Point(lp.toDouble(2, ok), lp.toDouble(3, ok), lp.toDouble(4, ok)));
          if(!ok) {
            App::log(tr("Error reading coordinates at line %1\n").arg(lineNumber));
          }

          event.setName(lp.toString(5, ok));
          if(!ok) {
            App::log(tr("Error reading name at line %1\n").arg(lineNumber));
          }

          QString typeS=lp.toString(6, ok);
          event.setType(SeismicEvent::convertType(typeS, ok));
          if(!ok) {
            App::log(tr("Error reading type at line %1\n").arg(lineNumber));
          }

          event.setForce(Point(lp.toDouble(7, ok), lp.toDouble(8, ok), lp.toDouble(9, ok)));
          if(!ok) {
            App::log(tr("Error reading force at line %1\n").arg(lineNumber));
          }
          if(ok) {
            events->add(event);
          }
        }
      }
      if(!ok) {
        Message::warning(MSG_ID, tr("Import seimsic events"),
                         tr("Error reading file '%1'. See log for details.")
                         .arg(fileName));
      }
      Engine::instance()->endSignalChange(model()->dataBase());
    }
  }
}

void SeismicEventView::exportEvents()
{
  TRACE;
  QList<SeismicEvent *> toExport=getSelection();
  if(toExport.isEmpty()) {
    return;
  }
  QString fileName=Message::getSaveFileName(tr("Export seismic events"),
                                            tr("Seismic events (*.evt)"));
  if(!fileName.isEmpty()) {
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      int n=toExport.count();
      for(int i=0; i<n; i++) {
        const SeismicEvent * event=toExport.at(i);
        s << event->time().toString(DateTime::defaultFormat) << " "
          << event->utmZone().toString() << " "
          << event->position().toString('g', 20) << " "
          << event->name() << " "
          << SeismicEvent::convertType(event->type()) << " "
          << event->force().toString('g', 20) << "\n";
      }
    } else {
      Message::warning(MSG_ID, tr("Export seimsic events"),
                       tr("Error writring to file '%1'.")
                       .arg(fileName));
    }
  }
}

