/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-07
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "XYColorPlot.h"
#include "XMLSciFigs.h"
#include "AxisWindow.h"
#include "GraphContentsLayerFactory.h"
#include "LayerPainterRequest.h"
#include "LayerLocker.h"

namespace SciFigs {

/*!
  \class XYColorPlot XYColorPlot.h
  \brief A XYColorPlot is a layer to draw dots of color (static)
*/

const QString XYColorPlot::xmlXYColorPlotTag="XYColorPlot";

REGISTER_GRAPHCONTENTLAYER(XYColorPlot, "XYColorPlot")

XYColorPlot::XYColorPlot(AxisWindow * parent) :
  XYPlot(parent) 
{
  TRACE;
    _pointColor=0;
}

XYColorPlot::~XYColorPlot()
{
  TRACE;
  delete _pointColor;
}

void XYColorPlot::setPointColors(VectorList<QColor> * c) 
{
  TRACE;
  LayerLocker ll(this);
  delete _pointColor;
  _pointColor=c;
}

void XYColorPlot::paintData(const LayerPainterRequest& lp, QPainter& p,double dotpercm) const
{
  TRACE;
  const GraphContentsOptions& gc=lp.options();
  int count;
  double pointSize, halfPointSize;
  getPaintParam(count, pointSize, halfPointSize, dotpercm);
  for(int i=0; i<count; i++) {
    if(lp.terminated()) return;
    double x=(*_xData)[i];
    if(x>gc.xVisMin() && x<gc.xVisMax()) {
      double y=(*_yData)[i];
      if(y>gc.yVisMin() && y<gc.yVisMax()) {
        p.setBrush((*_pointColor)[i]);
        p.setPen((*_pointColor)[i]);
        Symbol::paint(_symbol, p, gc.r2sF(x, y), pointSize, halfPointSize);
      }
    }
  }
}

void XYColorPlot::xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const
{
  TRACE;
  XYPlot::xml_writeBinaryData(s, context);
  VectorList<QColor>::iterator itc;
  for(itc=_pointColor->begin();itc!=_pointColor->end();++itc) {
    const QColor& c=*itc;
    s << (unsigned char)c.red();
    s << (unsigned char)c.green();
    s << (unsigned char)c.blue();
    s << (unsigned char)c.alpha();
  }
}

bool XYColorPlot::xml_setBinaryData(XML_SETBINARYDATA_ARGS)
{
  TRACE;
  if(XYPlot::xml_setBinaryData(s, context) && count()>0) {
    _pointColor=new VectorList<QColor>;
    _pointColor->resize(count());
    XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
    Version v(scifigsContext->currentFileVersion());
    static const Version vRef(2, 3, 0, "20091117");
    VectorList<QColor>::iterator itc;
    if(v>vRef) {
      unsigned char col[4];
      for(itc=_pointColor->begin();itc!=_pointColor->end();++itc) {
        s >> col[0];
        s >> col[1];
        s >> col[2];
        s >> col[3];
        (*itc).setRgb(col[0],col[1],col[2], col[3]);
      }
    } else {
      unsigned char col[3];
      for(itc=_pointColor->begin();itc!=_pointColor->end();++itc) {
        s >> col[0];
        s >> col[1];
        s >> col[2];
        (*itc).setRgb(col[0],col[1],col[2]);
      }
    }
  }
  return true;
}

} // namespace SciFigs
