/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-10-08
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Symbol.h"

namespace SciFigs {

/*!
  \class Symbol Symbol.h
  \brief A graphic symbol

*/

const QString Symbol::xmlSymbolTag="Symbol";
const Symbol Symbol::null;

Symbol::Symbol(Type type,double size, Pen pen, Brush brush)
{
  _type=type;
  _size=size<0 ? 0 : size;
  _pen=pen;
  _brush=brush;
}

bool Symbol::operator==(const Symbol& o) const
{
  TRACE;
  return _type==o._type
      && _size==o._size
      && _pen==o._pen
      && _brush==o._brush;
}

ENUM_AS_STRING_BEGIN(Symbol, Type)
ENUM_AS_STRING_DATA_8(Circle, Cross, Losange, NoSymbol, Square, Star, TriangleDown, TriangleUp);
ENUM_AS_STRING_END

void Symbol::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "type", convertType(_type));
  writeProperty(s, "size", _size);
}

void Symbol::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  _pen.xml_save(s, context);
  _brush.xml_save(s, context);
}

XMLMember Symbol::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="type") return XMLMember(0);
  else if(tag=="size") return XMLMember(1);
  else if(tag=="Pen") return XMLMember(&_pen);
  else if(tag=="Brush") return XMLMember(&_brush);
  return XMLMember(XMLMember::Unknown);
}

bool Symbol::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  bool ok=true;
  switch(memberID) {
  case 0:
    _type=convertType(content.toStringView(), ok);
    return ok;
  case 1:
    _size=content.toDouble(ok);
    return ok;
  default:
    return false;
  }
}

void Symbol::paint(QPainter& p, const QPointF& center, double dotpercm) const
{
  TRACE;
  double size=_size*dotpercm*0.1;
  p.save();
  p.setRenderHints(QPainter::Antialiasing, true);
  p.setBrush(_brush.qbrush(dotpercm) );
  p.setPen(_pen.qpen(dotpercm));
  paint(_type, p, center, size, 0.5*size);
  p.restore();
}

} // namespace SciFigs
