/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-03-23
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QtWidgets>

#include "SciFigsPlugin.h"

namespace SciFigs {

QStringList SciFigsPlugin::getPluginList(QStringList libs, bool& ok, bool debug)
{
  TRACE;
  QStringList plugins;
  int n=libs.count();
  if(n<=0) return plugins;
  QProgressDialog * d=new QProgressDialog(tr("Searching SciFigs plugins ..."),
                                          QString(), 0, n-1, QApplication::activeWindow());
  ok=true;
  for(int i=0;i<n;i++) {
    QString libFile=libs.at(i);
    if(!libFile.isEmpty()) {
	  if(debug) {
	    printf("Load library %s\n",libFile.toLatin1().data());
      fflush(stdout);
	  }
      SciFigsPlugin * pluginInstance=loadPlugin(libFile, debug);
      d->setValue(i);
      if(pluginInstance) {
        if( pluginInstance->isValid()) {
          if(debug) {
            printf("  SciFigs plugin interface\n");
            printf("  SciFigs interface version=%s\n",pluginInstance->interfaceVersion());
            fflush(stdout);
          }
          plugins << libFile;
        } else {
          App::log(tr("%1 is not compatible with current SciFigs version.\n").arg(libFile) );
          ok=false;
        }
      }
      CoreApplication::instance()->deletePlugin(pluginInstance);
    }
  }
  delete d;
  return plugins;
}

/*!
  Returns false if at least one plugin has not been built with a compatible SciFigs version or
  if no plugins can be loaded.
*/
bool SciFigsPlugin::loadPlugins(QStringList libs, bool debug)
{
  TRACE;
  bool correctInterfaceVersion=true;
  bool atLeastOne=false;
  for(QStringList::Iterator it=libs.begin(); it!=libs.end(); ++it) {
    if(!(*it).isEmpty()) {
      if(debug) {
        printf("Load library %s\n",it->toLatin1().data());
        fflush(stdout);
      }
      SciFigsPlugin * instance=loadPlugin(*it, debug);
      if(instance) {
        if(instance->isValid()) {
          if(debug) {
            printf("  SciFigs plugin interface\n");
            printf("  SciFigs interface version=%s\n",instance->interfaceVersion());
            fflush(stdout);
          }
          CoreApplication::instance()->addPlugin(instance);
          atLeastOne=true;
        } else {
          App::log(tr("%1 is not compatible with current SciFigs version.\n").arg(*it) );
          correctInterfaceVersion=false;
          CoreApplication::instance()->deletePlugin(instance);
        }
      }
    }
  }
  return correctInterfaceVersion && atLeastOne;
}

SciFigsPlugin * SciFigsPlugin::loadPlugin(QString pluginFile, bool debug)
{
  TRACE;
  QPluginLoader pluginTest;
  pluginTest.setFileName(pluginFile);
  QObject * instance=pluginTest.instance();
  if(instance) {
    if(debug) {
      printf("  Library loaded successfully\n");
      fflush(stdout);
    }
    SciFigsPlugin * pluginInstance=qobject_cast<SciFigsPlugin *>(instance);
    if(pluginInstance) {
      return pluginInstance;
    } else {
      delete instance;
    }
  } else if(debug) {
    printf("  %s\n",pluginTest.errorString().toLatin1().data());
    fflush(stdout);
  }
  return nullptr;
}

/*!
  Check if the plugin is compatible with SciFigs library
*/
bool SciFigsPlugin::isValid() const
{
  static const Version vRef(PackageInfo::package("SciFigs")->version());
  QString ver(interfaceVersion());
  Version v(ver);
  return v.compatible(vRef);
}

/*!
  \fn const char * SciFigsPlugin::interfaceVersion() const
  Returns the SciFigs version of the plugin (at compile time)
*/

} // namespace SciFigs
