/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-12-01
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>

#include "Pen.h"
#include "XMLSciFigs.h"

namespace SciFigs {

const QString Pen::xmlPenTag="Pen";
const Pen Pen::null;

void Pen::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  writeProperty(s, "color", color().name());
  writeProperty(s, "width", _width);
  writeProperty(s, "lineStyle", convertLineStyle(lineStyle()));
  writeProperty(s, "capStyle", convertCapStyle(capStyle()));
  writeProperty(s, "joinStyle", convertJoinStyle(joinStyle()));
}

XMLMember Pen::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  Q_UNUSED(attributes)
  Q_UNUSED(context)
  if(tag=="color") return XMLMember(0);
  else if(tag=="lineStyle") return XMLMember(1);
  else if(tag=="capStyle") return XMLMember(2);
  else if(tag=="joinStyle") return XMLMember(3);
  else if(tag=="width") return XMLMember(4);
  else if(tag=="style") return XMLMember(1);  // For compatibility
  return XMLMember(XMLMember::Unknown);
}

bool Pen::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  Q_UNUSED(tag)
  Q_UNUSED(attributes)
  bool ok=true;
  switch(memberID) {
  case 0: {
      Color c;
      c.fromString(content.toStringView(), ok);
      QColor qc;
      guiColor(c, qc);
      setColor(qc);
      return ok;
    }
  case 1:
    setLineStyle(convertLineStyle(content.toStringBuffer(), ok));
    return ok;
  case 2:
    setCapStyle(convertCapStyle(content.toStringBuffer(), ok));
    return ok;
  case 3:
    setJoinStyle(convertJoinStyle(content.toStringBuffer(), ok));
    return ok;
  case 4:
    _width=content.toDouble(ok);
    if(_width==0.0) {
      static const Version vRef(3, 3, 3);
      XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
      Version v(scifigsContext->currentFileVersion());
      if(v<vRef) {
        _width=0.1;
      }
    }
    return ok;
  default:
    return false;
  }
}

ENUM_AS_STRING_BEGIN(Pen, LineStyle)
ENUM_AS_STRING_DATA_17(Dash6DotDotLine, Dash6DotLine, Dash6Line, Dash6Space4Line,
                       Dash8DotDotLine, Dash8DotLine, Dash8Line, Dash8Space4Line,
                       DashDotDotLine, DashDotLine, DashLine, DashSpace4Line, DotLine,
                       DotSpace4Line, DotSpace8Line, NoLine, SolidLine);
ENUM_AS_STRING_SYNONYM("NoPen", NoLine);
ENUM_AS_STRING_DEFAULT_VALUE(SolidLine);
ENUM_AS_STRING_END

ENUM_AS_STRING_BEGIN(Pen, CapStyle)
ENUM_AS_STRING_DATA_3(FlatCap, RoundCap, SquareCap);
ENUM_AS_STRING_END

ENUM_AS_STRING_BEGIN(Pen, JoinStyle)
ENUM_AS_STRING_DATA_3(BevelJoin, MiterJoin, RoundJoin);
ENUM_AS_STRING_END

QPen Pen::qpen(double dotpercm) const
{
  TRACE;
  // By default SquareCap and BevelJoin
  QPen p(QBrush(_color), 0.1*dotpercm*_width, Qt::CustomDashLine);

  if(_width==0.0) {
    p.setStyle(Qt::NoPen);
    return p;
  }

  switch(_capStyle) {
  case SquareCap:
    break;
  case FlatCap:
    p.setCapStyle(Qt::FlatCap);
    break;
  case RoundCap:
    p.setCapStyle(Qt::RoundCap);
    break;
  }

  switch(_joinStyle) {
  case BevelJoin:
    break;
  case MiterJoin:
    p.setJoinStyle(Qt::MiterJoin);
    break;
  case RoundJoin:
    p.setJoinStyle(Qt::RoundJoin);
    break;
  }

  VectorList<qreal> pattern;
  switch(_lineStyle) {
  case NoLine:
  case LineStyleCount:
    p.setStyle(Qt::NoPen);
    return p;
  case SolidLine:
    p.setStyle(Qt::SolidLine);
    return p;
  case DashLine:
    pattern << 4 << 2;
    break;
  case Dash6Line:
    pattern << 6 << 2;
    break;
  case Dash8Line:
    pattern << 8 << 2;
    break;
  case DashSpace4Line:
    pattern << 4 << 4;
    break;
  case Dash6Space4Line:
    pattern << 6 << 4;
    break;
  case Dash8Space4Line:
    pattern << 8 << 4;
    break;
  case DotLine:
    pattern << 1 << 2;
    break;
  case DotSpace4Line:
    pattern << 1 << 4;
    break;
  case DotSpace8Line:
    pattern << 1 << 8;
    break;
  case DashDotLine:
    pattern << 4 << 2 << 1 << 2;
    break;
  case Dash6DotLine:
    pattern << 6 << 2 << 1 << 2;
    break;
  case Dash8DotLine:
    pattern << 8 << 2 << 1 << 2;
    break;
  case DashDotDotLine:
    pattern << 4 << 2 << 1 << 2 << 1 << 2;
    break;
  case Dash6DotDotLine:
    pattern << 6 << 2 << 1 << 2 << 1 << 2;
    break;
  case Dash8DotDotLine:
    pattern << 8 << 2 << 1 << 2 << 1 << 2;
    break;
  }
  p.setDashPattern(pattern);
  return p;
}

} // namespace SciFigs
