/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-05-11
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "LiveGridLayer.h"
#include "AutoAdjustScaleProperties.h"
#include "AxisWindow.h"
#include "LayerPainterRequest.h"
#include "IrregularGrid2DDraw.h"
#include "LayerLocker.h"
#include "XMLSciFigs.h"

namespace SciFigs {

LiveGridLayer::LiveGridLayer(AxisWindow * parent)
    : GridPlot(parent)
{
  TRACE;
  _function=nullptr;
  _functionValid=true;
  _sampling=1;
  connect(this, SIGNAL(paletteAdjusted(const QGpCoreMath::ColorMap&)),
          this, SLOT(showAdjustedPalette(const QGpCoreMath::ColorMap&)), Qt::QueuedConnection);
}

LiveGridLayer::~LiveGridLayer()
{
  TRACE;
  delete _function;
}

void LiveGridLayer::setFunction(FunctionPlot * f)
{
  TRACE;
  LayerLocker ll(this);
  if(f!=_function) {
    delete _function;
    _function=f;
  }
}

FunctionPlot * LiveGridLayer::takeFunction()
{
  TRACE;
  LayerLocker ll(this);
  FunctionPlot * f=_function;
  _function=nullptr;
  return f;
}

void LiveGridLayer::showAdjustedPalette(const ColorMap& map)
{
  TRACE;
  _colorMap=map;
}

void LiveGridLayer::adjustPalette(const IrregularGrid2D& grid, ColorMap& pal) const
{
  TRACE;
  double min, max;
  if(forceMinimumValue()) {
    min=minimumValue();
  } else {
    min=grid.minimumValue();
  }
  if(forceMaximumValue()) {
    max=maximumValue();
  } else {
    max=grid.maximumValue();
  }
  if(forceRatio()) {
    min=max/ratio();
  }
  if(symetric() && scale()==SamplingParameters::Linear) {
    if(fabs(max)>fabs(min)) {
      min=-max;
    } else {
      max=-min;
    }
  }
  pal.setValues(min, max, scale());
}

uint LiveGridLayer::_tab=PropertyTab::uniqueId();

/*!
  Setup property editor
*/
void LiveGridLayer::addProperties(PropertyProxy * pp)
{
  TRACE;
  GridPlot::addProperties(pp);
  if(pp->setCurrentTab(_tab)) {
    pp->addReference(this);
  } else {
    AutoAdjustScaleProperties * w=new AutoAdjustScaleProperties;
    pp->addTab(_tab, tr("Palette update"), w, this);
  }
}

/*!
  Clean property editor
*/
void LiveGridLayer::removeProperties(PropertyProxy * pp)
{
  TRACE;
  pp->removeTab(_tab, this);
  GridPlot::removeProperties(pp);
}

void LiveGridLayer::properties(PropertyWidget * w) const
{
  TRACE;
  if(w->id()==_tab) {
    AutoAdjustScale::properties(w);
  } else {
    GridPlot::properties(w);
  }
}

void LiveGridLayer::setProperty(uint wid, int pid, QVariant val)
{
  TRACE;
  if(wid==_tab) {
    AutoAdjustScale::setProperty(pid, val);
  } else {
    GridPlot::setProperty(wid, pid, val);
  }
}

Rect LiveGridLayer::boundingRect() const
{
  TRACE;
  return Rect();
}

void LiveGridLayer::paintData(const LayerPainterRequest& lp, QPainter& p, double) const
{
  TRACE;
  const QSize& size=lp.size();
  const GraphContentsOptions& gc=lp.options();

  GridIterator it(size, _sampling,
                  graph()->xAxis()->isEffectivelyReversed(),
                  graph()->yAxis()->isEffectivelyReversed());

  lp.graphContents()->setProgressMaximum(size.height());
  IrregularGrid2D * grid=calculateGrid(it, gc, &lp);
  lp.graphContents()->setProgressValue(size.height());
  if(!grid) {
    return;
  }

  // Make a copy of current palette (mandatory for multi-thread, palette are implicitly shared
  // objects so deep copy probably is not occuring right now, only upon user request for modification)
  ColorMap map=colorMap();
  if(autoAdjust()) {
    adjustPalette(*grid, map);
    emit paletteAdjusted(map);
  }
  IrregularGrid2DDraw d;
  // Get the limit of visible area in terms of grid indexes
  d.setVisibleArea(*grid, gc);
  // Create two int vectors containing the x and y of nodes (centers of blocks)
  d.setCellNodes(*grid, gc);

  if(smooth()) {
    drawGrid2DSmooth(*grid, map, lp, p, d);
  } else {
    d.setCellLimits();
    drawGrid2DBlock(*grid, map, lp, p, d);
  }
  delete grid;
  /*GraphContents * gc=graphContents();
  QImage im(w, h, QImage::Format_ARGB32);
  uint * imPtr=(uint * ) im.bits() + w + 1;
  double y;
  int h1=h-1;
  int w1=w-1;
  int iy;
  for(iy=1;iy<h1; iy+=3) {
    y=gc->ys2r(iy);
    fillRow(gc, imPtr, y, w, w1);
    imPtr+=2*w;
  }
  // Add last pixel rows
  switch(iy-h1) {
  case 0:  // two rows are missing
    y=gc->xs2r(iy);
    fillRowSup2(gc, imPtr, y, w, w1);
    break;
  case 1:  // one row is missing
    y=gc->xs2r(iy);
    fillRowSup1(gc, imPtr, y, w, w1);
    break;
  default: // h is a multiple of 3
    break;
  }
  p.drawImage(0, 0, im);*/
}

LiveGridLayer::GridIterator::GridIterator(const QSize& s, int sampling,
                                          bool xReversed, bool yReversed)
{
  _di=sampling;
  _nx=s.width()/_di;
  _ny=s.height()/_di;

  _iStart=_di >> 1;
  if(xReversed) {
    _ixGrid=_nx-1;
    _dixGrid=-1;
    _ixEndGrid=-1;
  } else {
    _ixGrid=0;
    _dixGrid=1;
    _ixEndGrid=_nx;
  }
  if(yReversed) {
    _iyGrid=0;
    _diyGrid=1;
    _iyEndGrid=_ny;
  } else {
    _iyGrid=_ny-1;
    _diyGrid=-1;
    _iyEndGrid=-1;
  }
}

IrregularGrid2D * LiveGridLayer::GridIterator::create(const GraphContentsOptions& gc) const
{
  IrregularGrid2D * grid=new IrregularGrid2D(_nx>1 ? _nx : 2, _ny>1 ? _ny : 2);
  int ix=_iStart;
  int ixGrid=_ixGrid;
  for(; ixGrid!=_ixEndGrid; ix+=_di, ixGrid+=_dixGrid) {
    grid->setX(ixGrid, gc.xs2r(ix));
  }
  int iy=_iStart;
  int iyGrid=_iyGrid;
  for(; iyGrid!=_iyEndGrid; iy+=_di, iyGrid+=_diyGrid) {
    grid->setY(iyGrid, gc.ys2r(iy));
  }
  return grid;
}

IrregularGrid2D * LiveGridLayer::calculateGrid(const GridIterator& it,
                                               const GraphContentsOptions& gc,
                                               const LayerPainterRequest * lp) const
{
  TRACE;
  if(!_function || !_function->function() || !_functionValid) {
    return nullptr;
  }
  IrregularGrid2D * grid=it.create(gc);

  double y;
  int ixGrid;
  int iy=it._iStart;
  int iyGrid=it._iyGrid;
  for(; iyGrid!=it._iyEndGrid; iy+=it._di, iyGrid+=it._diyGrid) {
    if(lp) {
      if(lp->terminated()) {
        delete grid;
        return nullptr;
      }
      lp->graphContents()->setProgressValue(iy);
    }
    ixGrid=it._ixGrid;
    y=grid->y(iyGrid);
    PrivateVector<double> vec(3, 0.0);
    PrivateVector<double> grad(3, 0.0);
    for(; ixGrid!=it._ixEndGrid; ixGrid+=it._dixGrid) {
      grid->setValue(ixGrid, iyGrid, _function->value(grid->x(ixGrid), y));
    }
  }
  return grid;
}

void LiveGridLayer::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  if(!_function || !_function->function() || !_functionValid) {
    return;
  }
  // Calculate grid with current plot limits
  GridIterator it(graphContents()->size(), _sampling,
                  graph()->xAxis()->isEffectivelyReversed(),
                  graph()->yAxis()->isEffectivelyReversed());
  IrregularGrid2D * grid=calculateGrid(it, graphContents()->options(), nullptr);

  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data()) {
    grid->xml_save(s, context);
  }
  delete grid;
  GridPlot::xml_writeChildren(s, context);
}

double LiveGridLayer::value(const Point2D& p) const
{
  TRACE;
  if(_function && _function->function() && _functionValid) {
    PrivateVector<double> vec(3, 0.0);
    PrivateVector<double> grad(3, 0.0);
    double val;
    lock(); // Value might not be thread-safe
    val=_function->value(p.x(), p.y());
    unlock();
    return val;
  } else {
    return std::numeric_limits<double>::quiet_NaN();
  }
}

QString LiveGridLayer::coordinateTipInfo(const Point2D&, const Point2D& pReal) const
{
  TRACE;
  static const QString str("(%2 ; %3 ; %1)");
  return str.arg(value(pReal));
}

} // namespace SciFigs
