/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-05-30
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef LINELAYER_H
#define LINELAYER_H

#include "GraphContentsLayer.h"
#include "SciFigsDLLExport.h"
#include "LineEditor.h"
#include "AbstractLine.h"
#include "Legend.h"

namespace SciFigs {

  class XMLSciFigs;

  class SCIFIGS_EXPORT LineLayer : public GraphContentsLayer
  {
    Q_OBJECT
    // Compatibility
    Q_PROPERTY(bool logErrorBars READ dummyPropertyInt WRITE setLogErrorBar)

    Q_PROPERTY(QString errorBar READ errorBarString WRITE setErrorBar)
    Q_PROPERTY(double signThreshold READ signThreshold WRITE setSignThreshold)
  public:
    LineLayer(AxisWindow * parent=nullptr);
    ~LineLayer();

    const QString& xml_tagName() const {return xmlLineLayerTag;}
    static const QString xmlLineLayerTag;

    void setReferenceLine(AbstractLine * l);
    void setReferencePen(const Pen& p);
    void setReferenceSymbol(const Symbol& s);

    void clear();
    void removeLine(int i);
    void removeLine(AbstractLine * l);
    void removeEmptyLines();

    AbstractLine * addLine();
    AbstractLine * addLine(const Pen& pen, const Symbol& sym);

    AbstractLine * line(int i) {return _lines.at(i);}
    const AbstractLine * line(int i) const {return _lines.at(i);}
    QList<AbstractLine *> lines() const;

    void setVisible(int i, bool v);
    void setSelected(int i, bool s);
    virtual Legend legend() const;

    // Return the number of curves
    int count() const {return _lines.count();}
    bool trackRectangle(int, double rx1, double ry1, double rx2, double ry2, Qt::KeyboardModifiers m);
    // Calculate the rectangle that includes all the curves
    Rect boundingRect() const;
    void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const;

    void mulYbyX(double constant);
    void divYbyX(double constant);

    enum ErrorBar {NoBar, VerticalBar, HorizontalBar, VerticalLogBar, HorizontalLogBar};
    void setLogErrorBar(bool b) {_errorBar=b ? VerticalLogBar : VerticalBar;} // Compatibility
    void setErrorBar(ErrorBar b) {_errorBar=b;}
    ErrorBar errorBar() const {return _errorBar;}
    void setErrorBar(QString b);
    QString errorBarString() const;

    double signThreshold() const {return _signThreshold;}
    void setSignThreshold(double t) {_signThreshold=t;}

    void setPointOptions(CurvePointOptions * o) {_pointOptions=o;}
    CurvePointOptions * pointOptions() {return _pointOptions;}
    const CurvePointOptions * pointOptions() const {return _pointOptions;}

    enum TrackingModes{Select=0, Pick, PickOrdered, Edit};
    bool mousePressEvent (QMouseEvent * e, int id=-1);
    bool mouseReleaseEvent (QMouseEvent * e, int id=-1);
    void mouseMoveEvent (const QPoint& pt, Qt::MouseButtons buttons, Qt::KeyboardModifiers modifiers);
    bool wheelEvent (QWheelEvent * e);
    bool keyPressEvent (QKeyEvent* e);
    void toggleTrackingAction(bool checked, int id=-1);

    typedef QList<AbstractLine *>::const_iterator const_iterator;
    const_iterator begin() const {return _lines.begin();}
    const_iterator end() const {return _lines.end();}

    bool hasProperties() {return true;}
    void addProperties(PropertyProxy * pp);
    void removeProperties(PropertyProxy * pp);
    void resetLineProperties();
    void properties(PropertyWidget * w) const;
    void setProperty(uint wid, int pid, QVariant val);

    bool canMerge(const GraphContentsLayer * o) const ;
    void merge(GraphContentsLayer * o);
  signals:
    void beginPointAdd(AbstractLine * line, int pointIndex);
    void endPointAdd(AbstractLine * line, int pointIndex);
    void pickLine(AbstractLine * line);
    void lineEdited(int lineIndex);
  public slots:
    void setLegend(const Legend& legend);
  protected:
    void xml_writeChildren(XML_WRITECHILDREN_ARGS) const;
    XMLMember xml_member(XML_MEMBER_ARGS);
    bool xml_setProperty(XML_SETPROPERTY_ARGS);

    virtual void drawCurve(const GraphContentsOptions& gc, QPainter& p, double dotpercm, int h,
                           Rect& r, const AbstractLine& line, int startAt, int endAt) const;

    int _currentLine;
    LineEditor * _lineEditor;
    ErrorBar _errorBar;
    double _signThreshold;
    CurvePointOptions * _pointOptions;
    AbstractLine * _referenceLine;
  private:
    void resetLayerProperties();

    QList<AbstractLine *> _lines;
    static uint _tabLineLayer;

    inline void drawVerticalErrorBar(double x, double minY, double maxY, const GraphContentsOptions& gc,
                                      QPainter& p, Rect& r, int tickLength) const;
    inline void drawHorizontalErrorBar(double y, double minX, double maxX, const GraphContentsOptions& gc,
                                        QPainter& p, Rect& r, int tickLength) const;
  };

  class LineLayerContext: public XMLContext
  {
  public:
    LineLayerContext(XMLSciFigs * parent, CurvePointOptions * options)
        : XMLContext(true)
    {
      _parent=parent;
      _pointOptions=options;
    }

    CurvePointOptions * options() const {return _pointOptions;}
    XMLSciFigs * parent() const {return _parent;}
  private:
    XMLSciFigs * _parent;
    CurvePointOptions * _pointOptions;
  };

} // namespace SciFigs

#endif // LINELAYER_H
