/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-03-19
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "LegendTableItem.h"

namespace SciFigs {

/*!
  \class LegendTableItem LegendTableItem.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
LegendTableItem::LegendTableItem(QObject * parent)
  : QAbstractTableModel(parent)
{
  _legend=0;
  _columnIndexes=0;
  _readOnlyText=false;
  _oneColor=true;
}

void LegendTableItem::setLegend(Legend * legend)
{
  TRACE;
  beginResetModel();
  _legend=legend;
  endResetModel();
}

void LegendTableItem::setColumnIndexes(const QHash<int, LegendTable::Column> * columnIndexes)
{
  TRACE;
  beginResetModel();
  _columnIndexes=columnIndexes;
  endResetModel();
}

inline LegendTable::Column LegendTableItem::column(int index) const
{
  QHash<int,LegendTable::Column>::const_iterator it=_columnIndexes->find(index);
  if(it!=_columnIndexes->end()) {
    return it.value();
  } else {
    return LegendTable::ColumnNull;
  }
}

int LegendTableItem::rowCount(const QModelIndex &parent) const
{
  TRACE;
  if( !parent.isValid() && _legend)
    return 2+_legend->count();
  return 2;
}

int LegendTableItem::columnCount(const QModelIndex & ) const
{
  return _columnIndexes ? _columnIndexes->count() : 0;
}

QVariant LegendTableItem::data(const QModelIndex &index, int role) const
{
  TRACE;
  if(!index.isValid()) return QVariant();
  switch (index.row()) {
  case 0:
    switch (role) {
    case Qt::DisplayRole:
      switch (column(index.column())) {
        case LegendTable::ColumnNull:
        case LegendTable::ColumnPenWidth:
        case LegendTable::ColumnPenStyle:
        case LegendTable::ColumnSymbolSize:
        case LegendTable::ColumnSymbolPenWidth:
        case LegendTable::ColumnSymbolPenStyle:
        case LegendTable::ColumnSymbolBrushStyle:
          break;
        case LegendTable::ColumnPenColor: return tr("Pen");
        case LegendTable::ColumnSymbolType: return tr("Symbol");
        case LegendTable::ColumnSymbolPenColor: return tr("Symbol pen");
        case LegendTable::ColumnSymbolBrushColor: return tr("Symbol brush");
        case LegendTable::ColumnText:
        case LegendTable::ColumnTextColor: return tr("Text");
      }
      return QVariant();
    case Qt::BackgroundRole:
      return Application::palette().button().color();
    case Qt::TextAlignmentRole:
      return (int) (Qt::AlignHCenter | Qt::AlignVCenter);
    default:
      return QVariant();
    }
    break;
  case 1:
    switch (role) {
    case Qt::DisplayRole:
      switch (column( index.column()) ) {
        case LegendTable::ColumnNull:
        case LegendTable::ColumnPenColor: return tr("Color");
        case LegendTable::ColumnPenWidth: return tr("Width");
        case LegendTable::ColumnPenStyle: return tr("Style");
        case LegendTable::ColumnSymbolType: return tr("Type");
        case LegendTable::ColumnSymbolSize: return tr("Size");
        case LegendTable::ColumnSymbolPenColor: return tr("Color");
        case LegendTable::ColumnSymbolPenWidth: return tr("Width");
        case LegendTable::ColumnSymbolPenStyle: return tr("Style");
        case LegendTable::ColumnSymbolBrushColor: return tr("Color");
        case LegendTable::ColumnSymbolBrushStyle: return tr("Style");
        case LegendTable::ColumnTextColor: return tr("Color");
        case LegendTable::ColumnText: return tr("Text");
      }
      return QVariant();
    case Qt::BackgroundRole:
      return QApplication::palette().button().color();
    case Qt::TextAlignmentRole:
      return (int) (Qt::AlignHCenter | Qt::AlignVCenter);
    default:
      return QVariant();
    }
    break;
  default: {
      int row=index.row()-2;
      switch (role) {
      case Qt::DisplayRole:
        switch (column( index.column()) ) {
        case LegendTable::ColumnNull: break;
        case LegendTable::ColumnPenColor:
          return _legend->pen(row).color();
        case LegendTable::ColumnPenWidth:
          return _legend->pen(row).width();
        case LegendTable::ColumnPenStyle:
          return LegendTable::penItem(_legend->pen(row).lineStyle());
        case LegendTable::ColumnSymbolType:
          return LegendTable::symbolItem(_legend->symbol(row).type());
        case LegendTable::ColumnSymbolSize:
          return _legend->symbol(row).size();
        case LegendTable::ColumnSymbolPenColor:
          return _legend->symbol(row).pen().color();
        case LegendTable::ColumnSymbolPenWidth:
          return _legend->symbol(row).pen().width();
        case LegendTable::ColumnSymbolPenStyle:
          return LegendTable::penItem(_legend->symbol(row).pen().lineStyle());
        case LegendTable::ColumnSymbolBrushColor:
          return _legend->symbol(row).brush().color();
        case LegendTable::ColumnSymbolBrushStyle:
          return LegendTable::brushItem(_legend->symbol(row).brush().style());
        case LegendTable::ColumnTextColor:
          return _legend->textColor(row);
        case LegendTable::ColumnText:
          return _legend->text(row);
        }
        return QVariant();
      case Qt::TextAlignmentRole:
        if(column( index.column())!=LegendTable::ColumnText) {
          return (int) (Qt::AlignHCenter | Qt::AlignVCenter);
        } else {
          return QVariant();
        }
      default:
        return QVariant();
      }
    }
    break;
  }
}

bool LegendTableItem::setData (const QModelIndex & index, const QVariant & value, int role)
{
  TRACE;
  if( !index.isValid() || index.row()<2) return false;
  int row=index.row()-2;
  switch (role) {
  case Qt::EditRole:
    switch (column( index.column()) ) {
    case LegendTable::ColumnNull: return false;
    case LegendTable::ColumnPenColor: {
        if(_oneColor) {
          setColor(row, value.value<QColor>());
        } else {
          Pen p=_legend->pen(row);
          p.setColor(value.value<QColor>());
          _legend->setPen(row, p);
        }
      }
      break;
    case LegendTable::ColumnPenWidth: {
        Pen p=_legend->pen(row);
        p.setWidth(value.toDouble());
        _legend->setPen(row, p);
      }
      break;
    case LegendTable::ColumnPenStyle: {
        Pen p=_legend->pen(row);
        p.setLineStyle(LegendTable::penStyle(value.toInt()));
        _legend->setPen(row, p);
      }
      break;
    case LegendTable::ColumnSymbolType: {
        Symbol s=_legend->symbol(row);
        s.setType(LegendTable::symbolType(value.toInt()));
        _legend->setSymbol(row, s);
      }
      break;
    case LegendTable::ColumnSymbolSize: {
        Symbol s=_legend->symbol(row);
        s.setSize(value.toDouble());
        _legend->setSymbol(row, s);
      }
      break;
    case LegendTable::ColumnSymbolPenColor: {
        if(_oneColor) {
          setColor(row, value.value<QColor>());
        } else {
          Symbol s=_legend->symbol(row);
          Pen p=s.pen();
          p.setColor(value.value<QColor>());
          s.setPen(p);
          _legend->setSymbol(row, s);
        }
      }
      break;
    case LegendTable::ColumnSymbolPenWidth: {
        Symbol s=_legend->symbol(row);
        Pen p=s.pen();
        p.setWidth(value.toDouble());
        s.setPen(p);
        _legend->setSymbol(row, s);
      }
      break;
    case LegendTable::ColumnSymbolPenStyle: {
        Symbol s=_legend->symbol(row);
        Pen p=s.pen();
        p.setLineStyle(LegendTable::penStyle(value.toInt()));
        s.setPen(p);
        _legend->setSymbol(row, s);
      }
      break;
    case LegendTable::ColumnSymbolBrushColor: {
        if(_oneColor) {
          setColor(row, value.value<QColor>());
        } else {
          Symbol s=_legend->symbol(row);
          Brush b=s.brush();
          b.setColor(value.value<QColor>());
          s.setBrush(b);
          _legend->setSymbol(row, s);
        }
      }
      break;
    case LegendTable::ColumnSymbolBrushStyle: {
        Symbol s=_legend->symbol(row);
        Brush b=s.brush();
        b.setStyle(LegendTable::brushStyle(value.toInt()) );
        s.setBrush(b);
        _legend->setSymbol(row, s);
      }
      break;
    case LegendTable::ColumnTextColor:
      if(_oneColor) {
        setColor(row, value.value<QColor>());
      } else {
        _legend->setTextColor(row, value.value<QColor>());
      }
      break;
    case LegendTable::ColumnText:
      _legend->setText(row, value.toString());
      break;
    }
    emit dataChanged(createIndex( row+2, 0), createIndex(row+2, columnCount()-1) );
    return true;
  default:
    return false;
  }
}

Qt::ItemFlags LegendTableItem::flags(const QModelIndex & index) const
{
  if(index.row()<2) {
    return Qt::ItemIsEnabled;
  } else {
    if(_readOnlyText && _columnIndexes->value(index.column())==LegendTable::ColumnText) {
      return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
    } else {
      switch(column(index.column())) {
      case LegendTable::ColumnNull:
      case LegendTable::ColumnText:
      case LegendTable::ColumnPenWidth:
      case LegendTable::ColumnSymbolSize:
      case LegendTable::ColumnSymbolPenWidth:
      case LegendTable::ColumnPenStyle:
      case LegendTable::ColumnSymbolPenStyle:
      case LegendTable::ColumnSymbolType:
      case LegendTable::ColumnSymbolBrushStyle:
        return Qt::ItemIsSelectable | Qt::ItemIsEditable | Qt::ItemIsEnabled;
        break;
      case LegendTable::ColumnPenColor:
      case LegendTable::ColumnSymbolPenColor:
      case LegendTable::ColumnSymbolBrushColor:
      case LegendTable::ColumnTextColor:
        break;
      }
      return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
    }
  }
}

void LegendTableItem::setColor(int row, const QColor& c)
{
  Pen p;
  Symbol s;
  Brush b;
  p=_legend->pen(row);
  p.setColor(c);
  _legend->setPen(row, p);
  s=_legend->symbol(row);
  p=s.pen();
  b=s.brush();
  p.setColor(c);
  b.setColor(c);
  s.setPen(p);
  s.setBrush(b);
  _legend->setSymbol(row, s);
  _legend->setTextColor(row, c);
}

} // namespace SciFigs
