/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-27
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "LegendTable.h"
#include "Symbol.h"
#include "LegendTableItem.h"
#include "LegendTableDelegate.h"
#include "SciFigsGlobal.h"

namespace SciFigs {

/*!
  \class LegendTable LegendTable.h
  \brief GUI table for Legend

*/

LegendTable::LegendTable(QWidget * parent)
   : TableView(parent)
{
  TRACE;
  LegendTableItem * model=new LegendTableItem(this);
  connect(model, SIGNAL(dataChanged (const QModelIndex &, const QModelIndex &)),
          this, SIGNAL(changed()));
  setModel(model);
  LegendTableDelegate * delegate=new LegendTableDelegate(this);
  setItemDelegate(delegate);
  connect(this, SIGNAL(clicked(const QModelIndex&)), delegate, SLOT(colorClicked(QModelIndex)));

  horizontalHeader()->setStretchLastSection(true);
  horizontalHeader()->hide();
  verticalHeader()->hide();
}

void LegendTable::setPropertySections(PropertySections s)
{
  TRACE;
  int columnIndex =0;
  _columnIndexes.clear();
  if(s & PenSection) {
    setSpan(0, columnIndex, 1, 3);
    _columnIndexes.insert(columnIndex++, ColumnPenColor);
    _columnIndexes.insert(columnIndex++, ColumnPenWidth);
    _columnIndexes.insert(columnIndex++, ColumnPenStyle);
  }
  if(s & SymbolTypeSection) {
    setSpan(0, columnIndex, 1, 2);
    _columnIndexes.insert(columnIndex++, ColumnSymbolType);
    _columnIndexes.insert(columnIndex++, ColumnSymbolSize);
  }
  if(s & SymbolPenSection) {
    setSpan(0, columnIndex, 1, 3);
    _columnIndexes.insert(columnIndex++, ColumnSymbolPenColor);
    _columnIndexes.insert(columnIndex++, ColumnSymbolPenWidth);
    _columnIndexes.insert(columnIndex++, ColumnSymbolPenStyle);
  }
  if(s & SymbolBrushSection) {
    setSpan(0, columnIndex, 1, 2);
    _columnIndexes.insert(columnIndex++, ColumnSymbolBrushColor);
    _columnIndexes.insert(columnIndex++, ColumnSymbolBrushStyle);
  }
  if(s & TextSection) {
    if(s & TextColorSection) {
      setSpan(0, columnIndex, 1, 2);
      _columnIndexes.insert(columnIndex++, ColumnTextColor);
      _columnIndexes.insert(columnIndex++, ColumnText);
    } else {
      setSpan(0, columnIndex, 2, 1);
      _columnIndexes.insert(columnIndex++, ColumnText);
    }
  } else if(s & TextColorSection) {
    setSpan(0, columnIndex, 2, 1);
    _columnIndexes.insert(columnIndex++, ColumnTextColor);
  }
  static_cast<LegendTableItem *>(model())->setColumnIndexes(&_columnIndexes);
  static_cast<LegendTableDelegate *>(itemDelegate())->setColumnIndexes(&_columnIndexes);
}

/*!
  Linked to properties: if \a c is true, the legend is the same for all
  legend widgets controled by this table. This is just a visual effect.

  TODO: find a good way to mark it on the screen
*/
void LegendTable::setConstant(bool c)
{
  QPalette palette=QWidget::palette();
  palette.setColor(QPalette::Text, c ? palette.color(QPalette::Text) : QColor(164,112,128));
  viewport()->setPalette(palette);
}

void LegendTable::setLegend(Legend * legend)
{
  static_cast<LegendTableItem *>(model())->setLegend(legend);
  resizeColumnsToContents();
  emit changed();
}

void LegendTable::setOneColor(bool oc)
{
  static_cast<LegendTableItem *>(model())->setOneColor(oc);
}

bool LegendTable::oneColor() const
{
  return static_cast<LegendTableItem *>(model())->oneColor();
}

void LegendTable::setReadOnlyText(bool ro)
{
  static_cast<LegendTableItem *>(model())->setReadOnlyText(ro);
}

bool LegendTable::readOnlyText() const
{
  return static_cast<LegendTableItem *>(model())->readOnlyText();
}

QPixmap LegendTable::lineStylePixmap(Pen::LineStyle s, const QPalette& palette)
{
  TRACE;
  QPixmap pm(32, 16);
  QPainter p;
  p.begin(&pm);
  p.fillRect(0, 0, 32, 16, palette.window().color());
  Pen pen(palette.windowText().color(), 0.3, s);
  p.setPen(pen.qpen(SciFigsGlobal::screenResolution()));
  p.drawLine(0, 8, 32, 8);
  p.end();
  return pm;
}

void LegendTable::initPenStyles(QComboBox * w)
{
  w->setIconSize(QSize(32, 16));
  for(int i=0; i<Pen::LineStyleCount; i++) {
    w->addItem(lineStylePixmap(static_cast<Pen::LineStyle>(i), w->palette()), "");
  }
}

Pen::LineStyle LegendTable::penStyle(int index)
{
  ASSERT(index>=0 && index<Pen::LineStyleCount);
  return static_cast<Pen::LineStyle>(index);
}

int LegendTable::penItem(Pen::LineStyle p)
{
  ASSERT(p<Pen::LineStyleCount);
  return static_cast<int>(p);
}

QPixmap LegendTable::symbolTypePixmap(Symbol::Type t, const QPalette& palette)
{
  TRACE;
  QPixmap pm(16, 16);
  QPainter p;
  p.begin(&pm);
  p.fillRect(0, 0, 16, 16, palette.window().color());
  Symbol s(t, 3.5, palette.windowText().color());
  s.paint(p, QPointF(7, 7));
  p.end();
  return pm;
}

void LegendTable::initSymbolTypes(QComboBox * w)
{
  w->setIconSize(QSize(16, 16));
  w->addItem(symbolTypePixmap(Symbol::NoSymbol, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::Cross, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::Square, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::Circle, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::TriangleUp, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::TriangleDown, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::Losange, w->palette()), "");
  w->addItem(symbolTypePixmap(Symbol::Star, w->palette()), "");
}

Symbol::Type LegendTable::symbolType(int index)
{
  switch (index) {
  default: return Symbol::NoSymbol;
  case 1: return Symbol::Cross;
  case 2: return Symbol::Square;
  case 3: return Symbol::Circle;
  case 4: return Symbol::TriangleUp;
  case 5: return Symbol::TriangleDown;
  case 6: return Symbol::Losange;
  case 7: return Symbol::Star;
  }
}

int LegendTable::symbolItem(Symbol::Type t)
{
  switch (t) {
  case Symbol::NoSymbol: break;
  case Symbol::Cross: return 1;
  case Symbol::Square: return 2;
  case Symbol::Circle: return 3;
  case Symbol::TriangleUp: return 4;
  case Symbol::TriangleDown: return 5;
  case Symbol::Losange: return 6;
  case Symbol::Star: return 7;
  }
  return 0;
}

QPixmap LegendTable::brushStylePixmap(Qt::BrushStyle s, const QPalette& palette)
{
  TRACE;
  QPixmap pm(16, 16);
  QPainter p;
  p.begin(&pm);
  p.fillRect(0, 0, 16, 16, palette.window().color());
  p.setPen(QPen(Qt::NoPen));
  p.setBrush(QBrush(palette.windowText().color(), s));
  p.drawRect(0, 0, 16, 16);
  p.end();
  return pm;
}

void LegendTable::initBrushStyles(QComboBox * w)
{
  w->setIconSize (QSize(16, 16));
  w->addItem(brushStylePixmap(Qt::NoBrush, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::SolidPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense1Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense2Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense3Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense4Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense5Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense6Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::Dense7Pattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::HorPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::VerPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::CrossPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::BDiagPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::FDiagPattern, w->palette()), "");
  w->addItem(brushStylePixmap(Qt::DiagCrossPattern, w->palette()), "");
}

Qt::BrushStyle LegendTable::brushStyle(int index)
{
  switch (index) {
  default: return Qt::NoBrush;
  case 1: return Qt::SolidPattern;
  case 2: return Qt::Dense1Pattern;
  case 3: return Qt::Dense2Pattern;
  case 4: return Qt::Dense3Pattern;
  case 5: return Qt::Dense4Pattern;
  case 6: return Qt::Dense5Pattern;
  case 7: return Qt::Dense6Pattern;
  case 8: return Qt::Dense7Pattern;
  case 9: return Qt::HorPattern;
  case 10: return Qt::VerPattern;
  case 11: return Qt::CrossPattern;
  case 12: return Qt::BDiagPattern;
  case 13: return Qt::FDiagPattern;
  case 14: return Qt::DiagCrossPattern;
  }
}

int LegendTable::brushItem(Qt::BrushStyle b)
{
  switch (b) {
  case Qt::LinearGradientPattern:
  case Qt::RadialGradientPattern:
  case Qt::ConicalGradientPattern:
  case Qt::TexturePattern:
  case Qt::NoBrush:
    break;
  case Qt::SolidPattern: return 1;
  case Qt::Dense1Pattern: return 2;
  case Qt::Dense2Pattern: return 3;
  case Qt::Dense3Pattern: return 4;
  case Qt::Dense4Pattern: return 5;
  case Qt::Dense5Pattern: return 6;
  case Qt::Dense6Pattern: return 7;
  case Qt::Dense7Pattern: return 8;
  case Qt::HorPattern: return 9;
  case Qt::VerPattern: return 10;
  case Qt::CrossPattern: return 11;
  case Qt::BDiagPattern: return 12;
  case Qt::FDiagPattern: return 13;
  case Qt::DiagCrossPattern: return 14;
  }
  return 0;
}

} // namespace SciFigs
