/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2015-02-26
**  Copyright: 2015-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ExportOptions.h"
#include "GraphicSheet.h"
#include "AxisWindow.h"
#include "LineLayer.h"
#include "XMLSciFigs.h"
#include "IrregularGrid2DPlot.h"

namespace SciFigs {

  /*!
    \class ExportOptions ExportOptions.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ExportOptions::ExportOptions()
  {
    // NO TRACE because used in initialization
    _dpi=0;  // No dpi default
    _preserveScales=false;
  }

  /*!
    Description of destructor still missing
  */
  ExportOptions::~ExportOptions()
  {
    TRACE;
  }

  bool ExportOptions::read(int& argc, char ** argv)
  {
    TRACE;
    int i, j=1;
    for(i=1; i<argc; i++) {
      QByteArray arg=argv[i];
      if(arg[0]=='-') {
        if(arg=="-export" || arg=="-e") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _exportFile=argv[i];
          _exportLayer=extractLayerName(_exportFile);
        } else if(arg=="-format" || arg=="-f") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _exportFormat=argv[i];
        } else if(arg=="-dpi") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _dpi=CoreApplication::toInt(i, i-1, argv);
          if(_dpi>1200) _dpi=1200;
          else if(_dpi<0) _dpi=0;
        } else if(arg=="-legend" || arg=="-l") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _legendFiles.append(argv[i]);
          _legendLayers.append(extractLayerName(_legendFiles.last()));
        } else if(arg=="-preserve-scales") {
          _preserveScales=true;
        } else if(arg=="-colormap" || arg=="-cm" || arg=="-p") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _colorMapFiles.append(argv[i]);
          _colorMapLayers.append(extractLayerName(_colorMapFiles.last()));
        } else if(arg=="-makeup" || arg=="-m") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _makeupObjectFiles.append(argv[i]);
        } else if(arg=="-select-objects") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _selectedObjects.append(QString(argv[i]).split(","));
        } else if(arg=="-append-layer" || arg=="-add-layer") { // Last test kept for compatibility
          CoreApplication::checkOptionArg(i, argc, argv);
          _appendLayerFiles.append(argv[i]);
        } else if(arg=="-prepend-layer") {
          CoreApplication::checkOptionArg(i, argc, argv);
          _prependLayerFiles.append(argv[i]);
        } else {
          argv[j++]=argv[i];
        }
      } else {
        argv[j++]=argv[i];
      }
    }
    if(j<argc) {
      argv[j]=nullptr;
      argc=j;
    }
    return true;
  }

  QString ExportOptions::extractLayerName(QString& file)
  {
    int index=file.indexOf('@');
    QString layer;
    if(index!=-1) {
      layer=file.mid(index+1);
      file=file.left(index);
    }
    return layer;
  }

  void ExportOptions::selectObjects(GraphicSheet * sheet) const
  {
    TRACE;
    if(_selectedObjects.isEmpty()) {
      sheet->selectAll();
    } else {
      sheet->selectObjects(_selectedObjects);
    }
  }

  QString ExportOptions::legendFile() const
  {
    return _legendFiles.isEmpty() ? QString() : _legendFiles.first();
  }

  QString ExportOptions::colorMapFile() const
  {
    return _colorMapFiles.isEmpty() ? QString() : _colorMapFiles.first();
  }

  bool ExportOptions::setLegend(GraphicSheet * sheet) const
  {
    TRACE;
    GraphicObjectList objects=sheet->selectedObjects();
    if(!_legendFiles.isEmpty()) {
      ASSERT(_legendLayers.count()==_legendFiles.count());
      for(int i=0; i<_legendFiles.count(); i++) {
        QList<GraphContentsLayer *> layerList;
        for(GraphicObjectList::iterator itw=objects.begin(); itw!=objects.end(); itw++) {
          AxisWindow * w=qobject_cast<AxisWindow *>(*itw);
          if(w && w->isSelected()) {
            layerList.append(w->graphContents()->findLayers<GraphContentsLayer>(_legendLayers.at(i)));
          }
        }
        if(!layerList.isEmpty()) {
          Legend legend;
          XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
          xmler.setTitle(tr("Load legend"));
          xmler.setFileName(_legendFiles.at(i));
          XMLSciFigs s;
          if(!xmler.exec(s.restoreFile(_legendFiles.at(i), &legend, XMLSciFigs::Data))) {
            return false;
          }
          for(QList<GraphContentsLayer *>::iterator itl=layerList.begin(); itl!=layerList.end(); itl++) {
            (*itl)->setLegend(legend);
          }
        }
      }
    }
    return true;
  }

  bool ExportOptions::setColorMap(GraphicSheet * sheet) const
  {
    TRACE;
    GraphicObjectList objects=sheet->selectedObjects();
    if(!_colorMapFiles.isEmpty()) {
      ASSERT(_colorMapLayers.count()==_colorMapFiles.count());
      for(int i=0; i<_colorMapFiles.count(); i++) {
        QList<GraphContentsLayer *> layerList;
        for(GraphicObjectList::iterator itw=objects.begin(); itw!=objects.end(); itw++) {
          AxisWindow * w=qobject_cast<AxisWindow *>(*itw);
          if(w && w->isSelected()) {
            layerList.append(w->graphContents()->findLayers<GraphContentsLayer>(_colorMapLayers.at(i)));
          }
        }
        if(!layerList.isEmpty()) {
          ColorMap colorMap;
          XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
          xmler.setTitle(tr("Load color map"));
          xmler.setFileName(_colorMapFiles.at(i));
          XMLSciFigs s;
          if(!xmler.exec(s.restoreFile(_colorMapFiles.at(i), &colorMap, XMLSciFigs::Data))) {
            return false;
          }
          for(QList<GraphContentsLayer *>::iterator itl=layerList.begin(); itl!=layerList.end(); itl++) {
            (*itl)->setColorMap(colorMap);
          }
        }
      }
    }
    return true;
  }

  /*!
    Returns the first color map ignoring the @LAYER_NAME tags.
  */
  ColorMap ExportOptions::firstColorMap() const
  {
    ColorMap m;
    if(_colorMapFiles.isEmpty()) {
      return m;
    }
    XMLErrorReport xmler(XMLErrorReport::Read | XMLErrorReport::NoMessageBox);
    xmler.setTitle(tr("Load color map"));
    xmler.setFileName(_colorMapFiles.first());
    XMLSciFigs s;
    if(!xmler.exec(s.restoreFile(_colorMapFiles.first(), &m, XMLSciFigs::Data))) {
      App::log(tr("Error reading the color map.\n"));
    }
    return m;
  }

  void ExportOptions::addLayers(GraphicSheet * sheet) const
  {
    TRACE;
    GraphicObjectList objects=sheet->selectedObjects();
    if(!_appendLayerFiles.isEmpty()) {
      for(GraphicObjectList::iterator itw=objects.begin(); itw!=objects.end(); itw++) {
        AxisWindow * w=qobject_cast<AxisWindow *>(*itw);
        if(w && w->isSelected()) {
          for(QStringList::const_iterator it=_appendLayerFiles.begin();
              it!=_appendLayerFiles.end(); it++) {
            w->graphContents()->appendLayers(*it);
          }
        }
      }
    }
    if(!_prependLayerFiles.isEmpty()) {
      for(GraphicObjectList::iterator itw=objects.begin(); itw!=objects.end(); itw++) {
        AxisWindow * w=qobject_cast<AxisWindow *>(*itw);
        if(w && w->isSelected()) {
          for(QStringList::const_iterator it=_prependLayerFiles.begin();
              it!=_prependLayerFiles.end(); it++) {
            w->graphContents()->prependLayers(*it);
          }
        }
      }
    }
  }

  void ExportOptions::restoreMakeUp(GraphicSheet * sheet) const
  {
    TRACE;
    foreach(QString m, _makeupObjectFiles) {
      sheet->restoreMakeUp(m);
    }
  }

  void ExportOptions::exportObjects(GraphicSheet * sheet) const
  {
    TRACE;
    if(!_exportFile.isEmpty()) {
      if(_selectedObjects.isEmpty() || _exportLayer=="all") {
        sheet->exportFile(_exportFile, _exportFormat, _exportLayer, _dpi);
      } else {
        GraphicSheet newSheet;
        newSheet.copyProperties(*sheet);
        newSheet.moveSelectionFrom(sheet);
        newSheet.exportFile(_exportFile, _exportFormat, _exportLayer, _dpi);
      }
    }
  }

  void ExportOptions::exportGridValues(GraphicSheet * sheet) const
  {
    TRACE;
    IrregularGrid2DPlot * layer=nullptr;
    QList<AxisWindow *> list=sheet->selectedObjects<AxisWindow>();
    for(int i=0; i<list.size(); i++) {
      GraphContents * gc=list.at(i)->graphContents();
      QList<IrregularGrid2DPlot *> layerList=gc->findLayers<IrregularGrid2DPlot>(QString());
      if(!layerList.isEmpty()) {
        layer=layerList.first();
        break;
      }
    }
    if(layer) {
      layer->exportValues(_exportFile);
    }
  }

  void ExportOptions::help(ApplicationHelp * h)
  {
    TRACE;
    h->addOption("-select-objects <OBJ1>[,<OBJ2>[,...]]", "Select the objects by their names. Applies to -export, "
                                                          "-append-layer and -prepend-layer.");
    h->addOption("-m, -makeup <FILE>", "Modifies the aspect of the object by the format specified in makeup file (.mkup). A "
                                       "graphicObject makeup file is expected. To get a template export it once with the gui "
                                       "from a context menu (not from the general menu bar which export sheet makeup files).\n"
                                       "Several occurences of this option is supported.");
    h->addOption("-e, -export <FILE[@LAYER_NAME]>", "Exports contents to FILE with format specified by option '-format' and exits. "
                                                    "If file has extension .layer (or format is set to LAYER) and LAYER_NAME is provided, "
                                                    "only the layer named LAYER_NAME is exported.");
    h->addOption("-f, -format <FORMAT>", "Specifies format for option '-export'\n"
                                          "  PAGE   Figue page (default)\n"
                                          "         Save sheet contents with an .page format.\n"
                                          "  LAYER  Figue layers\n"
                                          "         Save layers of plots with a .layer format. If -cpp is used and "
                                                   "various plot are created, a numbered extension _xxxx.layer will be added.\n"
                                          "  PDF    PDF document\n"
                                          "  SVG    Scalable Vector Graphics\n"
                                          "  BMP    Windows Bitmap\n"
                                          "  JPG    Joint Photographic Experts Group\n"
                                          "  JPEG   Joint Photographic Experts Group\n"
                                          "  PNG    Portable Network Graphics\n"
                                          "  PPM    Portable Pixmap\n"
                                          "  XBM    X11 Bitmap\n"
                                          "  XPM    X11 Pixmap");
    h->addOption("-dpi <DPI>", "Forces resolution to DPI(dot per inch) in export file");
    h->addOption("-l, -legend <FILE[@LAYER_NAME]>","Legend to set style and color to lines. If LAYER_NAME is provided, "
                                                   "only this layer is modified.");
    h->addOption("-cm, -colormap <FILE[@LAYER_NAME]>","Color map to set colors. If LAYER_NAME is provided, "
                                                      "only this layer is modified.");
    h->addOption("-preserve-scales","Preserve scales for x, y and color map (default=automatically adjust).");
    h->addOption("-append-layer <FILE>", "Adds FILE layer on top of the current layer stack. "
                                         "Several occurences can be used to add more than one layer file.");
    h->addOption("-prepend-layer <FILE>", "Adds FILE layer at the bottom of the current layer stack. "
                                          "Several occurences can be used to add more than one layer file.");
  }


} // namespace SciFigs
