/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-02-21
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "CircleMask.h"
#include "LayerPainterRequest.h"
#include "GraphContentsLayerFactory.h"

namespace SciFigs {

  /*!
    \class CircleMask CircleMask.h
    \brief A circular mask

    Mask all inferior layers and leave only a visible circle.

    TODO: add a property widget
  */

  const QString CircleMask::xmlCircleMaskTag="CircleMask";

  REGISTER_GRAPHCONTENTLAYER(CircleMask, "CircleMask")

  /*!
    Constructs a circle centered around 0,0 with a radius of 1.
  */
  CircleMask::CircleMask(AxisWindow * parent)
      : CircleViewer(parent)
  {
    TRACE;
    add(0.0, 0.0, 1.0, 1.0, 0.0, Pen(Qt::white));
  }

  /*!
    Sets circle position, circle sizes and mask color.
  */
  void CircleMask::set(double x, double y, double rx, double ry, QColor col)
  {
    TRACE;
    CircleViewer::set(0, x, y, rx, ry, 0.0, col);
  }

  /*!
    Paints the mask.
  */
  void CircleMask::paintData(const LayerPainterRequest& lp, QPainter& p, double) const
  {
    TRACE;
    const GraphContentsOptions& gc=lp.options();
    Limits limits(gc);
    double startPhi, stopPhi;
    const Item& item=_items[0];
    const QGpCoreMath::Ellipse& ell=item._ellipse;
    if(limits.polarLimits(ell, startPhi, stopPhi)) {
      QPainterPath path;
      Rect r(gc.xr2s(gc.xVisMin()), gc.yr2s(gc.yVisMin()), gc.xr2s(gc.xVisMax()), gc.yr2s(gc.yVisMax()));
      path.addRect((int)r.x1(), (int)r.y1(), (int)r.width(), (int)r.height());
      int rx=abs((int)(ell.majorRadius()*gc.ax()));
      int ry=abs((int)(ell.minorRadius()*gc.ay()));
      const Point2D& c=ell.center();
      int orx=gc.xr2s(c.x())-rx;
      int ory=gc.yr2s(c.y())-ry;
      int minphi=(int)floor(Angle::radiansToDegrees(startPhi)*16.0);
      int dphi=(int)ceil(Angle::radiansToDegrees(stopPhi)*16.0)-minphi;
      path.arcTo(orx, ory, rx + rx, ry + ry, minphi, dphi);
      p.setPen(QPen(Qt::NoPen));
      p.setBrush(item._pen.color());
      p.drawPath(path);
    }
  }

  /*!
    Returns bounding rectangle.
  */
  Rect CircleMask::boundingRect() const
  {
    TRACE;
    const QGpCoreMath::Ellipse& ell=_items[0]._ellipse;
    const Point2D& c=ell.center();
    double xRadius=ell.xRadius();
    double yRadius=ell.yRadius();
    Rect r(c.x()-xRadius, c.y()-yRadius, c.x()+xRadius, c.y()+yRadius);
    return r;
  }

  /*!
    Makes sure that there is only one circle.
  */
  bool CircleMask::xml_polish(XML_POLISH_ARGS)
  {
    TRACE;
    Q_UNUSED(context)
    while(count()>1) {
      remove(0);
    }
    return true;
  }

} // namespace SciFigs
