/***************************************************************************
**
**  This file is part of QGpGuiStat.
**
**  QGpGuiStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpGuiStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-10-27
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SampleClassification.h"
#include "ui_SampleClassification.h"

namespace QGpGuiStat {

  /*
   *  Constructs a SampleClassification as a child of 'parent', with the
   *  name 'name' and widget flags set to 'f'.
   *
   *  The dialog will by default be modeless, unless you set 'modal' to
   *  true to construct a modal dialog.
   */
  SampleClassification::SampleClassification(QWidget *parent, Qt::WindowFlags f)
      : Dialog(parent, f),
        _ui(new Ui::SampleClassification)
  {
    TRACE;
    _ui->setupUi(this);
    connect(_ui->absoluteThresholdCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->relativeThresholdCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->curveRangeCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->curveRangeType, SIGNAL(currentIndexChanged(int)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->aboveCurveCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->belowCurveCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->insideCurvesCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
    connect(_ui->outsideCurvesCondition, SIGNAL(toggled(bool)), this, SLOT(enableValueConditionWidgets()));
  }

  void SampleClassification::setCurveList(const QList<CurvePropertiesWidget *>& list)
  {
    TRACE;
    for(QList<CurvePropertiesWidget *>::const_iterator it=list.begin(); it!=list.end(); it++) {
      _ui->curve1Name->addItem((*it)->proxy()->name());
      _ui->curve2Name->addItem((*it)->proxy()->name());
    }
    if(_ui->curve1Name->count()<2) {
      _ui->insideCurvesCondition->setEnabled(false);
      _ui->insideCurvesCondition->setChecked(false);
      _ui->outsideCurvesCondition->setEnabled(false);
      _ui->outsideCurvesCondition->setChecked(false);
      if(_ui->curve1Name->count()<1) {
        _ui->curveRangeCondition->setEnabled(false);
        _ui->curveRangeCondition->setChecked(false);
        _ui->aboveCurveCondition->setEnabled(false);
        _ui->aboveCurveCondition->setChecked(false);
        _ui->belowCurveCondition->setEnabled(false);
        _ui->belowCurveCondition->setChecked(false);
      }
    }
  }

  void SampleClassification::setCategories(const Legend * cat)
  {
    if(cat) {
      for(int i=0; i<cat->count(); i++) {
        _ui->categories->addItem(cat->text(i));
      }
      _ui->categoryCondition->setCategories(*cat);
    } else {
      delete _ui->categoryConditionGroup;
      _ui->categoryConditionGroup=nullptr;
      _ui->categoryCondition=nullptr;
      delete _ui->categorizeAction;
      _ui->categorizeAction=nullptr;
      delete _ui->categories;
      _ui->categories=nullptr;
    }
  }

  void SampleClassification::on_categorizeAction_toggled(bool checked)
  {
    _ui->categories->setEnabled(checked);
  }

  void SampleClassification::setUnchecked(QRadioButton * activeBut, QRadioButton * uncheckBut)
  {
    if(activeBut!=uncheckBut) {
      uncheckBut->blockSignals(true);
      uncheckBut->setChecked(false);
      uncheckBut->blockSignals(false);
    }
  }

  void SampleClassification::enableValueConditionWidgets()
  {
    TRACE;
    // De-check all other options
    QRadioButton * but=qobject_cast<QRadioButton *>(sender());
    if(but) {
      setUnchecked(but, _ui->absoluteThresholdCondition);
      setUnchecked(but, _ui->relativeThresholdCondition);
      setUnchecked(but, _ui->curveRangeCondition);
      setUnchecked(but, _ui->insideCurvesCondition);
      setUnchecked(but, _ui->outsideCurvesCondition);
      setUnchecked(but, _ui->aboveCurveCondition);
      setUnchecked(but, _ui->belowCurveCondition);
    }
    bool b;
    b=_ui->absoluteThresholdCondition->isChecked() && _ui->absoluteThresholdCondition->isEnabled();
    _ui->absoluteThresholdValue->setEnabled(b);
    b=_ui->relativeThresholdCondition->isChecked() && _ui->relativeThresholdCondition->isEnabled();
    _ui->relativeThresholdValue->setEnabled(b);

    b=_ui->curveRangeCondition->isChecked() && _ui->curveRangeCondition->isEnabled();
    bool twoCurves=(_ui->insideCurvesCondition->isChecked() && _ui->insideCurvesCondition->isEnabled()) ||
                   (_ui->outsideCurvesCondition->isChecked() && _ui->outsideCurvesCondition->isEnabled());
    bool anyCurve=b ||
                  (_ui->aboveCurveCondition->isChecked() && _ui->aboveCurveCondition->isEnabled()) ||
                  (_ui->belowCurveCondition->isChecked() && _ui->belowCurveCondition->isEnabled()) ||
                  twoCurves;
    _ui->curve1Name->setEnabled(anyCurve);
    _ui->curve2Label->setEnabled(twoCurves);
    _ui->curve2Name->setEnabled(twoCurves);
    _ui->curveRangeType->setEnabled(b);
    _ui->curveRangeFactor->setEnabled(b);
    switch(_ui->curveRangeType->currentIndex()) {
    default:
      _ui->curveRangeFactor->setSuffix("");
      break;
    case 1:
    case 4:
      _ui->curveRangeFactor->setSuffix(" %");
      break;
    case 2:
    case 5:
      _ui->curveRangeFactor->setSuffix(" sigma");
      break;
    }
    if(twoCurves) {
      if(_ui->curve1Name->currentIndex()==_ui->curve2Name->currentIndex()) {
        int i=_ui->curve1Name->currentIndex()+1;
        if(i<_ui->curve1Name->count()) {
          _ui->curve2Name->setCurrentIndex(i);
        } else {
          _ui->curve2Name->setCurrentIndex(0);
        }
      }
      if(_ui->curve1Name->currentIndex()>_ui->curve2Name->currentIndex()) {
        int i1=_ui->curve1Name->currentIndex();
        int i2=_ui->curve2Name->currentIndex();
        _ui->curve1Name->setCurrentIndex(i2);
        _ui->curve2Name->setCurrentIndex(i1);
      }
    }
  }

  void SampleClassification::updateAllFields()
  {
    enableValueConditionWidgets();
    if(_ui->categoryCondition) {
      _ui->categoryCondition->updateAllFields();
    }
  }

  void SampleClassification::on_curve1Name_activated(int)
  {
    TRACE;
    if(_ui->curve1Name->currentIndex()==_ui->curve2Name->currentIndex()) {
      int i=_ui->curve1Name->currentIndex()+1;
      if(i>=_ui->curve2Name->count()) {
        i-=2;
      }
      _ui->curve2Name->setCurrentIndex(i);
    }
  }

  void SampleClassification::on_curve2Name_activated(int)
  {
    TRACE;
    if(_ui->curve1Name->currentIndex()==_ui->curve2Name->currentIndex()) {
      int i=_ui->curve2Name->currentIndex()+1;
      if(i>=_ui->curve1Name->count()) {
        i-=2;
      }
      _ui->curve1Name->setCurrentIndex(i);
    }
  }

  /*!

  */
  bool SampleClassification::run(CurveBrowser * curves,
                                 const Legend * categories,
                                 Histogram2D * hist)
  {
    TRACE;
    SampleClassification * d=new SampleClassification(Application::instance()->activeWindow());
    d->setCurveList(curves->curves());
    d->setCategories(categories);
    Settings::getWidget(d);
    d->enableValueConditionWidgets();
    if(d->exec()==QDialog::Accepted) {
      Settings::setWidget(d);
      SampleClassificationParameters param;
      CurveProxy * proxy1=nullptr;
      CurveProxy * proxy2=nullptr;

      if(d->_ui->resetAction->isChecked()) {
        param.setAction(SampleClassificationParameters::Reset);
      } else if(d->_ui->validateAction->isChecked()) {
        param.setAction(SampleClassificationParameters::Validate);
      } else if(d->_ui->invalidateAction->isChecked()) {
        param.setAction(SampleClassificationParameters::Invalidate);
      } else if(d->_ui->categorizeAction && d->_ui->categorizeAction->isChecked()) {
        param.setAction(SampleClassificationParameters::Categorize);
        param.setCategory(d->_ui->categories->currentIndex());
      }

      if(d->_ui->categoryCondition) {
        param.setCategoryConditionType(d->_ui->categoryCondition->type());
        param.setCategoryCondition(d->_ui->categoryCondition->category());
      } else {
        param.setCategoryConditionType(SampleClassificationParameters::AllCategories);
      }

      param.setValueConditionType(SampleClassificationParameters::NoCondition);
      if(d->_ui->absoluteThresholdCondition->isChecked()) {
        param.setValueConditionType(SampleClassificationParameters::AbsoluteThreshold);
        param.setThreshold(d->_ui->absoluteThresholdValue->value());
      } else if(d->_ui->relativeThresholdCondition->isChecked()) {
        param.setValueConditionType(SampleClassificationParameters::RelativeThreshold);
        param.setThreshold(d->_ui->relativeThresholdValue->value()*0.01);
      } else if(d->_ui->curveRangeCondition->isChecked()) {
        int index=d->_ui->curve1Name->currentIndex();
        if(index>=0) {
          proxy1=curves->curveAt(index)->proxy();
        }
        switch(d->_ui->curveRangeType->currentIndex()) {
        default:
          param.setValueConditionType(SampleClassificationParameters::CurveAbsoluteInsideRange);
          param.setAbsoluteRange(d->_ui->curveRangeFactor->value());
          break;
        case 1:
          param.setValueConditionType(SampleClassificationParameters::CurveRelativeInsideRange);
          param.setRelativeRange(d->_ui->curveRangeFactor->value()*0.01);
          break;
        case 2:
          param.setValueConditionType(SampleClassificationParameters::CurveStddevInsideRange);
          param.setStddevRange(d->_ui->curveRangeFactor->value());
          break;
        case 3:
          param.setValueConditionType(SampleClassificationParameters::CurveAbsoluteOutsideRange);
          param.setAbsoluteRange(d->_ui->curveRangeFactor->value());
          break;
        case 4:
          param.setValueConditionType(SampleClassificationParameters::CurveRelativeOutsideRange);
          param.setRelativeRange(d->_ui->curveRangeFactor->value()*0.01);
          break;
        case 5:
          param.setValueConditionType(SampleClassificationParameters::CurveStddevOutsideRange);
          param.setStddevRange(d->_ui->curveRangeFactor->value());
          break;
        }
      } else if(d->_ui->aboveCurveCondition->isChecked()) {
        int index=d->_ui->curve1Name->currentIndex();
        if(index>=0) {
          proxy1=curves->curveAt(index)->proxy();
          param.setValueConditionType(SampleClassificationParameters::AboveCurve);
        }
      } else if(d->_ui->belowCurveCondition->isChecked()) {
        int index=d->_ui->curve1Name->currentIndex();
        if(index>=0) {
          proxy1=curves->curveAt(index)->proxy();
          param.setValueConditionType(SampleClassificationParameters::BelowCurve);
        }
      } else if(d->_ui->insideCurvesCondition->isChecked()) {
        int index1=d->_ui->curve1Name->currentIndex();
        int index2=d->_ui->curve2Name->currentIndex();
        if(index1>=0 && index2>=0) {
          proxy1=curves->curveAt(index1)->proxy();
          proxy2=curves->curveAt(index2)->proxy();
          param.setValueConditionType(SampleClassificationParameters::InsideCurves);
        }
      } else if(d->_ui->outsideCurvesCondition->isChecked()) {
        int index1=d->_ui->curve1Name->currentIndex();
        int index2=d->_ui->curve2Name->currentIndex();
        if(index1>=0 && index2>=0) {
          proxy1=curves->curveAt(index1)->proxy();
          proxy2=curves->curveAt(index2)->proxy();
          param.setValueConditionType(SampleClassificationParameters::OutsideCurves);
        }
      }
      delete d;
      hist->classifySamples(param, proxy1, proxy2);
      return true;
    } else {
      delete d;
      return false;
    }
  }

} // namespace QGpGuiStat
