/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-11-25
**  Copyright: 2011-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "WindowFunctionParameterWidget.h"

namespace QGpGuiMath {

  /*!
    \class WindowFunctionParameterWidget WindowFunctionParameterWidget.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  WindowFunctionParameterWidget::WindowFunctionParameterWidget(QWidget * parent)
    : QWidget(parent)
  {
    TRACE;
    setupUi(this);
    windowTypeEdit->installEventFilter(this);

    connect(parameterEdit, SIGNAL(valueChanged(double)), this, SIGNAL(parametersChanged()));
  }

  bool WindowFunctionParameterWidget::eventFilter(QObject * obj, QEvent * e)
  {
    TRACE;
    switch (e->type()) {
    case QEvent::WhatsThisClicked:
      QDesktopServices::openUrl(QUrl(static_cast<QWhatsThisClickedEvent *>(e)->href()));
      e->accept();
      return true;
    default:
      return QWidget::eventFilter(obj, e);
    }
  }

  void WindowFunctionParameterWidget::updateAllFields()
  {
    TRACE;
    on_windowTypeEdit_activated(0);
  }

  void WindowFunctionParameterWidget::on_windowTypeEdit_activated(int)
  {
    TRACE;
    switch(windowTypeEdit->currentIndex()) {
    case 8: // Blackman
      parameterEdit->show();
      parameterLabel->show();
      parameterLabel->setText(tr("Alpha"));
      parameterEdit->setSuffix(QString());
      parameterEdit->setMaximum(1);
      parameterEdit->setSingleStep(0.1);
      parameterEdit->setWhatsThis(QString());
      break;
    case 13:  // Gaussian
      parameterEdit->show();
      parameterLabel->show();
      parameterLabel->setText(tr("Width (sigma)"));
      parameterEdit->setSuffix(QString());
      parameterEdit->setMaximum(0.5);
      parameterEdit->setSingleStep(0.01);
      parameterEdit->setWhatsThis(QString());
      break;
    case 14:  // Tukey
      parameterEdit->show();
      parameterLabel->show();
      parameterLabel->setText(tr("Width (alpha/2)"));
      parameterEdit->setSuffix(" %");
      parameterEdit->setMaximum(100.0);
      parameterEdit->setSingleStep(5.0);
      parameterEdit->setWhatsThis(tr("One side taper relative width."));
      break;
    default: // For others, no parameters
      parameterLabel->hide();
      parameterEdit->hide();
      break;
    }
    emit parametersChanged();
  }

  void WindowFunctionParameterWidget::getParameters(WindowFunctionParameters& param) const
  {
    TRACE;
    switch(windowTypeEdit->currentIndex()) {
    case 0:
      param.setShape(WindowFunctionParameters::Rectangular);
      break;
    case 1:
      param.setShape(WindowFunctionParameters::Bartlett);
      break;
    case 2:
      param.setShape(WindowFunctionParameters::Triangular);
      break;
    case 3:
      param.setShape(WindowFunctionParameters::Parzen);
      break;
    case 4:
      param.setShape(WindowFunctionParameters::Welch);
      break;
    case 5:
      param.setShape(WindowFunctionParameters::Cosine);
      break;
    case 6:
      param.setShape(WindowFunctionParameters::Hann);
      break;
    case 7:
      param.setShape(WindowFunctionParameters::Hamming);
      break;
    case 8:
      param.setShape(WindowFunctionParameters::Blackman);
      param.setAlpha(parameterEdit->value());
      break;
    case 9:
      param.setShape(WindowFunctionParameters::Nuttall);
      break;
    case 10:
      param.setShape(WindowFunctionParameters::BlackmanNuttall);
      break;
    case 11:
      param.setShape(WindowFunctionParameters::BlackmanHarris);
      break;
    case 12:
      param.setShape(WindowFunctionParameters::FlatTop);
      break;
    case 13:
      param.setShape(WindowFunctionParameters::Gaussian);
      param.setSigma(parameterEdit->value());
      break;
    case 14:
      param.setShape(WindowFunctionParameters::Tukey);
      param.setAlpha(parameterEdit->value()*0.02);
      break;
    case 15:
      param.setShape(WindowFunctionParameters::BartlettHann);
      break;
    case 16:
      param.setShape(WindowFunctionParameters::Lanczos);
      break;
    case 17:
      param.setShape(WindowFunctionParameters::KonnoOhmachi);
      break;
    default:
      ASSERT(false);
      break;
    }
  }

  void WindowFunctionParameterWidget::setParameters(const WindowFunctionParameters& param)
  {
    TRACE;
    switch(param.shape()) {
    case WindowFunctionParameters::Rectangular:
      windowTypeEdit->setCurrentIndex(0);
      break;
    case WindowFunctionParameters::Bartlett:
      windowTypeEdit->setCurrentIndex(1);
      break;
    case WindowFunctionParameters::Triangular:
      windowTypeEdit->setCurrentIndex(2);
      break;
    case WindowFunctionParameters::Parzen:
      windowTypeEdit->setCurrentIndex(3);
      break;
    case WindowFunctionParameters::Welch:
      windowTypeEdit->setCurrentIndex(4);
      break;
    case WindowFunctionParameters::Cosine:
      windowTypeEdit->setCurrentIndex(5);
      break;
    case WindowFunctionParameters::Hann:
      windowTypeEdit->setCurrentIndex(6);
      break;
    case WindowFunctionParameters::Hamming:
      windowTypeEdit->setCurrentIndex(7);
      break;
    case WindowFunctionParameters::Blackman:
      windowTypeEdit->setCurrentIndex(8);
      parameterEdit->setValue(param.alpha());
      break;
    case WindowFunctionParameters::Nuttall:
      windowTypeEdit->setCurrentIndex(9);
      break;
    case WindowFunctionParameters::BlackmanNuttall:
      windowTypeEdit->setCurrentIndex(10);
      break;
    case WindowFunctionParameters::BlackmanHarris:
      windowTypeEdit->setCurrentIndex(11);
      break;
    case WindowFunctionParameters::FlatTop:
      windowTypeEdit->setCurrentIndex(12);
      break;
    case WindowFunctionParameters::Gaussian:
      windowTypeEdit->setCurrentIndex(13);
      parameterEdit->setValue(param.sigma());
      break;
    case WindowFunctionParameters::Tukey:
      windowTypeEdit->setCurrentIndex(14);
      parameterEdit->setValue(50.0*param.alpha());
      break;
    case WindowFunctionParameters::BartlettHann:
      windowTypeEdit->setCurrentIndex(15);
      break;
    case WindowFunctionParameters::Lanczos:
      windowTypeEdit->setCurrentIndex(16);
      break;
    case WindowFunctionParameters::KonnoOhmachi:
      windowTypeEdit->setCurrentIndex(17);
      break;
    }
  }

} // namespace QGpGuiMath
