/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-09-13
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SamplingParameterWidget.h"

namespace QGpGuiMath {

  /*!
    \class SamplingParameterWidget SamplingParameterWidget.h
    \brief Widget to modify a SamplingParameters

    Maintains an internal SamplingParameters to provide correct transition
    from step to count mode.
  */
  SamplingParameterWidget::SamplingParameterWidget(QWidget* parent)
      : QWidget(parent)
  {
    TRACE;
    setupUi(this);

    _admissibleMin=-std::numeric_limits<double>::infinity();
    _admissibleMax=std::numeric_limits<double>::infinity();
  }

  void SamplingParameterWidget::setRangeOnly()
  {
    TRACE;
    delete scaleTypeLabel;
    scaleTypeLabel=nullptr;
    delete scaleTypeEdit;
    scaleTypeEdit=nullptr;
    delete stepCountOption;
    stepCountOption=nullptr;
    delete stepCountValue;
    stepCountValue=nullptr;
  }

  void SamplingParameterWidget::updateAllFields()
  {
    TRACE;
    // Set scale of param without changing step/count value
    switch(scaleTypeEdit->currentIndex()) {
    case 0:
      _param.setScaleType(SamplingParameters::Log, true);
      break;
    case 1:
      _param.setScaleType(SamplingParameters::Linear, true);
      break;
    case 2:
      _param.setScaleType(SamplingParameters::Inverse, true);
      break;
    }
    _param.setRange(fromValue->value(), toValue->value());
    on_stepCountValue_valueChanged(stepCountValue->value());
    // Set again the scale type to adjust admissible ranges
    on_scaleTypeEdit_currentIndexChanged(scaleTypeEdit->currentIndex());
  }

  void SamplingParameterWidget::setPrecision(int digit)
  {
    TRACE;
    fromValue->setDecimals(digit);
    toValue->setDecimals(digit);
  }

  void SamplingParameterWidget::setSingleStep(double s)
  {
    TRACE;
    fromValue->setSingleStep(s);
    toValue->setSingleStep(s);
  }

  void SamplingParameterWidget::setAdmissibleRange(double min, double max)
  {
    TRACE;
    if(min<max) {
      _admissibleMin=min;
      _admissibleMax=max;
    } else {
      _admissibleMin=max;
      _admissibleMax=min;
    }
    fromValue->setMinimum(_admissibleMin);
    toValue->setMinimum(_admissibleMin);
    fromValue->setMaximum(_admissibleMax);
    toValue->setMaximum(_admissibleMax);
    if(scaleTypeEdit) {
      if(_admissibleMax<0.0) {
        scaleTypeEdit->setCurrentIndex(1);
        scaleTypeEdit->setEnabled(false);
      } else {
        scaleTypeEdit->setEnabled(true);
      }
      if(scaleTypeEdit->currentIndex()==0 && _admissibleMin<0.0) {
        fromValue->setMinimum(pow(10, -fromValue->decimals()) );
        toValue->setMinimum(pow(10, -fromValue->decimals()) );
      }
    }
  }

  void SamplingParameterWidget::on_fromValue_valueChanged(double val)
  {
    TRACE;
    _param.setMinimum(val);
    parametersChanged();
  }

  void SamplingParameterWidget::on_toValue_valueChanged(double val)
  {
    TRACE;
    _param.setMaximum(val);
    parametersChanged();
  }

  void SamplingParameterWidget::on_scaleTypeEdit_currentIndexChanged(int index)
  {
    TRACE;
    switch(index) {
    case 0:
      if(fromValue->minimum()<=0.0) {
        fromValue->setMinimum(pow(10, -fromValue->decimals()));
      }
      if(toValue->minimum()<=0.0) {
        toValue->setMinimum(pow(10, -fromValue->decimals()));
      }
      _param.setScaleType(SamplingParameters::Log, true);
      break;
    case 1:
      fromValue->setMinimum(_admissibleMin);
      toValue->setMinimum(_admissibleMin);
      _param.setScaleType(SamplingParameters::Linear, true);
      break;
    case 2:
      if(fromValue->minimum()<=0.0) {
        fromValue->setMinimum(pow(10, -fromValue->decimals()));
      }
      if(toValue->minimum()<=0.0) {
        toValue->setMinimum(pow(10, -fromValue->decimals()));
      }
      _param.setScaleType(SamplingParameters::Inverse, true);
      break;
    }
    on_stepCountOption_currentIndexChanged(stepCountOption->currentIndex());
  }

  void SamplingParameterWidget::on_stepCountOption_currentIndexChanged(int index)
  {
    TRACE;
    stepCountValue->blockSignals(true);
    if(index==0) {
      double minStep, delta;
      int ndec;
      switch(scaleTypeEdit->currentIndex()) {
      default:
        delta=_param.maximum()/_param.minimum();
        minStep=::pow(delta, 1.0/static_cast<double>(10000-1));
        ndec=1-qFloor(::log10(minStep-1.0));
        break;
      case 1:
        delta=_param.maximum()-_param.minimum();
        minStep=delta/static_cast<double>(10000-1);
        ndec=1-qFloor(::log10(minStep));
        break;
      case 2:
        delta=1.0/_param.minimum()-1.0/_param.maximum();
        minStep=delta/static_cast<double>(10000-1);
        ndec=1-qFloor(::log10(minStep));
        break;
      }
      stepCountValue->setDecimals(ndec<0 ? 0 : ndec);
      stepCountValue->setMinimum(minStep);
      stepCountValue->setMaximum(delta);
      stepCountValue->setSingleStep(::pow(10.0, ::floor(::log10(_param.step()-1.0)))-1.0);
      stepCountValue->setValue(_param.step());
    } else {
      stepCountValue->setDecimals(0);
      stepCountValue->setMinimum(1.0);
      stepCountValue->setMaximum(10000.0);
      stepCountValue->setValue(_param.count());
      stepCountValue->setSingleStep(1.0);
    }
    //_param.setType(oldscaleType);
    stepCountValue->blockSignals(false);
    emit parametersChanged();
  }

  void SamplingParameterWidget::on_stepCountValue_valueChanged(double val)
  {
    TRACE;
    if(stepCountOption->currentIndex()==0) {
      _param.setStep(val);
    } else {
      _param.setCount(qRound(val));
    }
    emit parametersChanged();
  }

  void SamplingParameterWidget::setParameters(const SamplingParameters & param)
  {
    TRACE;
    _param=param;
    fromValue->setValue(param.minimum());
    toValue->setValue(param.maximum());
    if(scaleTypeEdit) {
      switch(param.scaleType()) {
      case SamplingParameters::Log:
        scaleTypeEdit->setCurrentIndex(0);
        break;
      case SamplingParameters::Linear:
        scaleTypeEdit->setCurrentIndex(1);
        break;
      case SamplingParameters::Inverse:
        scaleTypeEdit->setCurrentIndex(2);
        break;
      }
      stepCountOption->setCurrentIndex(param.stepType()==SamplingParameters::Step ? 0 : 1);
    }
  }

  void SamplingParameterWidget::setFrequency()
  {
    TRACE;
    setUnit(tr(" Hz"));
    setPrecision(2);
    setSingleStep(0.25);
    setAdmissibleRange(1e-99, std::numeric_limits<double>::infinity());
  }

} // namespace QGpGuiMath
