/***************************************************************************
**
**  This file is part of QGpCoreWave.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "TheoreticalFK.h"

namespace QGpCoreWave {

  /*!
    \class TheoreticalFK TheoreticalFK.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  TheoreticalFK::TheoreticalFK(const VectorList<Point2D>& stations)
  {
    _stations=stations;
    _fac=_stations.count();
    _fac=1.0/(_fac*_fac);
  }

  double TheoreticalFK::value(Vector<double>& k) const
  {
    Complex q;
    Complex B;
    for(int i=_stations.count()-1; i>=0; i--) {
      // Minus sign skipped... q or q*, it does not matter because abs2() is calculated.
      q.setUnitExp(k.scalarProduct(_stations[i]));
      B+=q;
    }
    return B.abs2()*_fac;
  }

  double TheoreticalFK::radialFirstDerivative(const Vector<double>& k) const
  {
    int n=_stations.count();
    // Theta is from 0 to 180 deg.
    double theta=k.azimuth(0, 1);
    if(theta>M_PI) {
      theta-=M_PI;
    }
    double ctheta=cos(theta);
    double stheta=sin(theta);
    // Get station coordinates projected on theta direction
    DoubleMatrix r(n, 1);
    for(int i=n-1; i>=0; i--) {
      r.at(i)=ctheta*_stations[i].x()+stheta*_stations[i].y();
    }
    ComplexMatrix q(n, 1);
    ComplexMatrix rq(1, n);
    for(int i=n-1; i>=0; i--) {
      Complex& qi=q.at(i);
      Complex& rqi=rq.at(i);
      rqi.setUnitExp(k.scalarProduct(_stations[i]));
      qi=rqi;
      qi.conjugate();
      rqi*=r.at(i);
    }
    ComplexMatrix f=q*rq;
    double b=0.0;
    Complex * fvalues=f.values();
    for(int i=n*n-1; i>=0; i--) {
      b+=fvalues[i].im();
    }
    return -2.0*b*_fac;
  }

  double TheoreticalFK::radialSecondDerivative(const Vector<double>& k) const
  {
    int n=_stations.count();
    // Theta is from 0 to 180 deg.
    double theta=k.azimuth(0, 1);
    if(theta>M_PI) {
      theta-=M_PI;
    }
    double ctheta=cos(theta);
    double stheta=sin(theta);
    // Get station coordinates projected on theta direction
    DoubleMatrix r(n, 1);
    for(int i=n-1; i>=0; i--) {
      r.at(i)=ctheta*_stations[i].x()+stheta*_stations[i].y();
    }
    ComplexMatrix q(n, 1);
    ComplexMatrix rq(n, 1);
    ComplexMatrix r2q(1, n);
    for(int i=n-1; i>=0; i--) {
      Complex& qi=q.at(i);
      Complex& rqi=rq.at(i);
      Complex& r2qi=r2q.at(i);
      double& ri=r.at(i);
      qi.setUnitExp(k.scalarProduct(_stations[i]));
      rqi=qi;
      rqi*=ri;
      r2qi=qi;
      r2qi*=ri*ri;
      r2qi.conjugate();
    }
    ComplexMatrix f1=q*r2q;
    ComplexMatrix f2=rq*rq.conjugateTransposedVector();
    double b=0.0;
    Complex * f1values=f1.values();
    Complex * f2values=f2.values();
    for(int i=n*n-1; i>=0; i--) {
      b+=f1values[i].re()-f2values[i].re();
    }
    return -2.0*b*_fac;
  }

  double TheoreticalFK::radialSecondDerivativeAtOrigin(const Vector<double>& k) const
  {
    int n=_stations.count();
    // Theta is from 0 to 180 deg.
    double theta=k.azimuth(0, 1);
    if(theta>M_PI) {
      theta-=M_PI;
    }
    double ctheta=cos(theta);
    double stheta=sin(theta);
    // Get station coordinates projected on theta direction
    DoubleMatrix r(n, 1);
    for(int i=n-1; i>=0; i--) {
      r.at(i)=ctheta*_stations[i].x()+stheta*_stations[i].y();
    }
    double b=0.0;
    for(int i=0; i<n; i++) {
      for(int j=0; j<n; j++) {
        b+=r.at(i)*r.at(j);
      }
      b-=n*r.at(i)*r.at(i);
    }
    return 2.0*b*_fac;
  }

  /*!
    Get \a min and \a max distances in array coordinates
  */
  void TheoreticalFK::distanceRange(double& min, double& max) const
  {
    const VectorList<Point2D>& s=stations();
    int i=s.count()-1;
    int j=i-1;
    if(j<0) { // Less than 2 stations in array
      min=0.0;
      max=0.0;
      return;
    }
    double d=s.at(i).squareDistanceTo(s.at(j));
    min=d;
    max=d;
    for(; i>=0; i--) {
      const Point2D& p1=s.at(i);
      for(j--; j>=0; j--) {
        const Point2D& p2=s.at(j);
        d=p1.squareDistanceTo(p2);
        if(d<min) {
          min=d;
        } else if(d>max) {
          max=d;
        }
      }
    }
    min=::sqrt(min);
    max=::sqrt(max);
  }

  /*!
    Calculate a linear regression over the array coordinates.
    Get the rotated rectangle that includes all points.
    Return the ratio between width and height which is an indicator of the array shape.
  */
  double TheoreticalFK::shapeFactor() const
  {
    const VectorList<Point2D>& s=stations();
    int n=s.count();
    LinearRegression lr;
    for(int i=n-1; i>=0; i--) {
      const Point2D& p=s.at(i);
      lr.add(p.x(), p.y());
    }
    Angle rot;
    if(lr.calculate()) {
      rot.set(1.0, lr.a());
    } else {
      rot.set(0.0, 1.0);
    }
    Rect r;
    for(int i=n-1; i>=0; i--) {
      Point2D p=s.at(i);
      p.rotate(rot);
      r.add(p.x(), p.y());
    }
    return r.width()/r.height();
  }

} // namespace QGpCoreWave
