/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  QGpCoreStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-11
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HistogramInversion.h"
#include "GaussianMixtureLikelihood.h"

namespace QGpCoreStat {

  /*!
    \class HistogramInversion HistogramInversion.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  HistogramInversion::HistogramInversion(int histogramCount, QObject * parent)
    : QObject(parent)
  {
    TRACE;
    ASSERT(histogramCount>=1);
    _histogramCount=histogramCount;
    _histograms=new const Histogram2D*[_histogramCount];
    for(int i=0; i<_histogramCount; i++) {
      _histograms[i]=nullptr;
    }
    _histogram=nullptr;

    _currentIndex=-1;
    _stopIndex=0;
    _na=nullptr;
    _distributions=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  HistogramInversion::~HistogramInversion()
  {
    TRACE;
    if(_na) {
      _na->stop();
      _na->wait();
      delete _na;
    }
    delete _histogram;
    delete [] _histograms;
    delete [] _distributions;
  }

  /*!
    All histograms must have the same X sampling.
  */
  void HistogramInversion::setHistogram(int index, const Histogram2D * h)
  {
    TRACE;
    if(!_distributions) {
      _distributions=new GaussianMixtureDistribution[h->nx()];
    }
    _histograms[index]=h;
  }

  bool HistogramInversion::checkHistograms() const
  {
    for(int i=0; i<_histogramCount; i++) {
      if(!_histograms[i]) {
        App::log(tr("Histogram %1 is not set\n").arg(i));
        return false;
      }
    }
    int nx=_histograms[0]->nx();
    for(int i=1; i<_histogramCount; i++) {
      if(nx!=_histograms[i]->nx()) {
        App::log(tr("Histogram %1 has not %2 X values\n").arg(i).arg(_histograms[0]->nx()));
        return false;
      }
    }
    for(int ix=0; ix<nx; ix++) {
      VectorList<bool> fp=_histograms[0]->validSamplesFootprint(ix);
      for(int i=1; i<_histogramCount; i++) {
        if(fp!=_histograms[i]->validSamplesFootprint(ix)) {
          App::log(tr("Histogram %1 has not the same set of valid samples as the first one\n").arg(i));
          return false;
        }
      }
    }
    return true;
  }

  void HistogramInversion::start()
  {
    TRACE;
    _currentIndex=-1;
    _stopIndex=_histograms[0]->nx();
    if(checkHistograms()) {
      internalStart();
    }
  }

  void HistogramInversion::start(int index)
  {
    TRACE;
    if(index<0) {
      index=0;
    }
    _stopIndex=index+1;
    if(_stopIndex>_histograms[0]->nx()) {
      _stopIndex=_histograms[0]->nx();
    }
    _currentIndex=index-1;
    if(checkHistograms()) {
      internalStart();
    }
  }

  void HistogramInversion::internalStart()
  {
    TRACE;
    if(_na) {
      _na->stop();
      _na->wait();
      delete _na;
      _na=nullptr;
    }

    _na=new ModelRepository;
    GaussianMixtureLikelihood forward(&_parameters, _histogramCount);
    do {
      // Find next non-empty histogram
      int sampleCount=0;
      do {
        _currentIndex++;
        if(_currentIndex==_stopIndex) {
          App::freeze(false);
          delete _na;
          _na=nullptr;
          emit finished();
          return;
        }

        sampleCount=_histograms[0]->validSampleCount(_currentIndex);
        App::freeze(false);
        App::log(1, tr("      %1/%2 at x=%3 found %4 samples\n")
                         .arg(_currentIndex+1)
                         .arg(_histograms[0]->nx())
                         .arg(_histograms[0]->x(_currentIndex))
                         .arg(sampleCount));
        App::freeze(!_parameters.verbose());
      } while(sampleCount==0);

      delete _histogram;
      _histogram=new MultivariateHistogram(_histogramCount);
      for(int id=0; id<_histogramCount; id++) {
        _histogram->setSampling(id, _histograms[id]->ySamplingParameters());
      }
      PrivateVector<double> x(_histogramCount);
      const Histogram2D& hist0=*_histograms[0];
      for(int is=_histograms[0]->sampleCount()-1; is>=0; is--) {
        const Histogram2D::Sample& s=hist0.sample(is);
        if(s.isValid() && hist0.indexOfX(s.x())==_currentIndex) {
          for(int id=0; id<_histogramCount; id++) {
            const Histogram2D& hist=*_histograms[id];
            if(hist.ySampling() & LogScale) {
              x[id]=log(hist.sample(is).y());
            } else if(hist.ySampling() & InverseScale) {
              x[id]=1.0/hist.sample(is).y();
            } else {
              x[id]=hist.sample(is).y();
            }
          }
          _histogram->addSample(x);
        }
      }
      // Remove buckets with only one value
      _histogram->removeSmallBucketValues(1.0);
      _histogram->normalize();
      forward.setHistogram(_histogram);
    } while(!forward.setInversionParameters() ||
            !_na->setForward(&forward));

    QString reportFileName="/tmp/gaussmix.report";
    ReportWriter::initReport(reportFileName, "", ReportWriter::Overwrite);
    _na->openReport(reportFileName);

    _na->setStorage();
    _na->setStableMisfitCount(5000);
    _na->setSignificantMisfitReduction(0.05);
    _na->setBestModelCount(50);
    _na->setMaximumModelCount(50);
    connect(_na, SIGNAL(finished()), this, SLOT(monteCarloFinished()));
    _na->start(INT_MAX, Generator::MonteCarlo);
  }

  void HistogramInversion::wait()
  {
    TRACE;
    while(_currentIndex<_histograms[0]->nx()) {
      CoreApplication::processEvents(QEventLoop::AllEvents, 1000);
    }
  }

  void HistogramInversion::monteCarloFinished()
  {
    TRACE;
    disconnect(_na, SIGNAL(finished()), this, SLOT(monteCarloFinished()));
    _na->setMaximumModelCount(50000);
    connect(_na, SIGNAL(finished()), this, SLOT(neighborhoodFinished()));
    _na->start(INT_MAX, Generator::Neighborhood);
  }

  void HistogramInversion::neighborhoodFinished()
  {
    TRACE;
    _na->closeReport();
    disconnect(_na, SIGNAL(finished()), this, SLOT(neighborhoodFinished()));
    SetIndex bestIndex=_na->bestModelIndex();
    if(bestIndex.isValid()) {
      GaussianMixtureLikelihood * f=static_cast<GaussianMixtureLikelihood *>(_na->forward());
      f->setValues(_na->model(bestIndex));
      _distributions[_currentIndex]=f->distribution();
    }
    emit histogramFinished(_currentIndex);

    internalStart();
  }

  /*!
    Returns the solution for \a index as a human readable string.
  */
  QString HistogramInversion::humanSolution(int index, const QString& prefix)
  {
    TRACE;
    const GaussianMixtureDistribution& gmd=_distributions[index];
    QString s, sm;
    const QString tm("%1%2*[%3]\n"), td("%1(%2)");
    int nd=gmd.dimensionCount();
    for(int im=0; im<gmd.modeCount(); im++) {
      const MultivariateNormalDistribution& d=gmd.mode(im);
      sm=tm.arg(prefix).arg(gmd.weight(im));
      QString sd;
      for(int id=0; id<nd; id++) {
        if(id>0) {
          sd+=", ";
        }
        SamplingOptions samp=_histograms[id]->ySampling();
        if(samp & LogScale) {
          sd+=td.arg(exp(d.mean(id))).arg(exp(d.stddev(id)));
        } else if(samp & InverseScale) {
          sd+=td.arg(1.0/d.mean(id)).arg(1.0/d.stddev(id));
        } else {
          sd+=td.arg(d.mean(id)).arg(d.stddev(id));
        }
      }
      s+=sm.arg(sd);
    }
    return s;
  }

} // namespace QGpCoreStat

