/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-12-18
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreMath.h>

#include "Histogram2D.h"

namespace QGpCoreStat {

  /*!
    Read and parse next available not blank and not starting with '#',
    and containing \a pattern.
  */
  bool Histogram2D::Sample::read(const QString& line, int xColumn, int yColumn, int validColumn)
  {
    TRACE;
    bool ok=true;
    LineParser lp(line);
    setX(lp.toDouble(xColumn, ok));
    setY(lp.toDouble(yColumn, ok));
    if(validColumn>=0) {
      _valid=(lp.toInt(validColumn, ok)==1);
    } else {
      _valid=true;
    }
    _line=line;
    if(ok) {
      return true;
    } else {
      // Double \n to avoid overwrite by progress value
      App::log(tr("Error parsing line '%1'\n\n").arg(line) );
      return false;
    }
  }

  void Histogram2D::Sample::writeData(QTextStream& s) const
  {
    TRACE;
    if(_valid) {
      s << toString('g', 20) << "\n";
    }
  }

  void Histogram2D::Sample::writeLine(QTextStream& s) const
  {
    TRACE;
    if(_valid) {
      s << _line << "\n";
    }
  }

  /*!
    \class Histogram2D Histogram2D.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  Histogram2D::Histogram2D(int nx, int ny)
      : IrregularGrid2D(nx, ny)
  {
    TRACE;
    _xSampling=LinearScale;
    _ySampling=LinearScale;
    _periodic=false;
    init(0.0);
  }

  Histogram2D::Histogram2D(const Histogram2D& o)
      : IrregularGrid2D(o)
  {
    TRACE;
    _xSampling=o._xSampling;
    _ySampling=o._ySampling;
    _xTitle=o._xTitle;
    _yTitle=o._yTitle;
    _samples=o._samples;
    _periodic=o._periodic;
  }

  /*!
    Description of destructor still missing
  */
  Histogram2D::~Histogram2D()
  {
    TRACE;
  }

  bool Histogram2D::setXSampling(const SamplingParameters& s)
  {
    TRACE;
    _xSampling=SamplingParameters::samplingOptions(s.scaleType());
    if(s.isValid()) {
      for(int i=0; i<nx(); i++) {
        setX(i, s.value(i));
      }
      return true;
    } else {
      App::log(tr("Bad min/max for x sampling\n") );
      return false;
    }
  }


  bool Histogram2D::setXSampling(const VectorList<double>& x)
  {
    TRACE;
    if(x.count()>0 && x.first()<=x.last()) {
      for(int i=0; i<nx(); i++) {
        setX(i, x.at(i));
      }
      return true;
    } else {
      App::log(tr("Bad x sampling\n") );
      return false;
    }
  }

  bool Histogram2D::setYSampling(const SamplingParameters& s)
  {
    TRACE;
    _ySampling=SamplingParameters::samplingOptions(s.scaleType());
    if(_periodic) {
      _ySampling=LinearScale;
    }
    if(std::isfinite(s.minimum()) &&
       std::isfinite(s.maximum()) &&
       s.minimum()<s.maximum()) {
      for(int i=0; i<ny(); i++) {
        setY(i, s.value(i));
      }
      return true;
    } else {
      App::log(tr("Bad min/max for y sampling\n") );
      return false;
    }
  }

  /*!
    Set Y values as periodic. The period is equal to max-min.
  */
  void Histogram2D::setPeriodic(bool p)
  {
    _periodic=p;
    if(_periodic) {
      _ySampling=LinearScale;
    }
  }

  double Histogram2D::period() const
  {
    // The first and the last class of a periodic value are equivalent.
    // It is an artificial split of the same class.
    // The period is not exactly the difference between the max and the min.
    return y(ny()-1)-y(1);
  }

  /*!
    Returns false if \a s is empty.
  */
  bool Histogram2D::setSamples(const VectorList<Sample>& s)
  {
    TRACE;
    if(s.isEmpty()) {
      return false;
    } else {
      _samples=s;
      countSamples();
      return true;
    }
  }

  void Histogram2D::addSample(const Point2D &s)
  {
    TRACE;
    int ix=indexOfX(s.x());
    int iy=indexOfY(s.y());
    if(ix>-1 && ix<nx() && iy>-1 && iy<ny()) {
      (*valuePointer(ix, iy))++;
    }
    _samples.append(Sample(s));
  }

  /*!
    Count samples on a scaled axis, limits of classes are influenced by the transformation.
  */
  void Histogram2D::countSamples(IrregularGrid2D * grid) const
  {
    TRACE;
    if(_ySampling & LogScale) {
      grid->log(YAxis);
    } else if(_ySampling & InverseScale) {
      grid->inverse(YAxis);
    }

    int stat=0;
    grid->init(0.0); // Reset all hit count to 0
    int ix, iy;
    if(nx()==1) {
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples.at(i);
        if(s.isValid()) {
          if(_ySampling & LogScale) {
            iy=grid->indexOfY(::log(s.y()));
          } else if(_ySampling & InverseScale) {
            iy=grid->indexOfY(1.0/s.y());
          } else {
            iy=grid->indexOfY(s.y());
          }
          if(iy>-1 && iy<ny()) {
            //ASSERT(left(ix)<=s.x() && s.x()<=right(ix));
            //ASSERT(bottom(iy)<=s.y() && s.y()<=top(iy));
            (*grid->valuePointer(0, iy))++;
          }
        }
      }
    } else {
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples.at(i);
        if(s.isValid()) {
          ix=grid->indexOfX(s.x());
          if(ix>-1 && ix<nx()) {
            if(_ySampling & LogScale) {
              iy=grid->indexOfY(::log(s.y()));
            } else if(_ySampling & InverseScale) {
              iy=grid->indexOfY(1.0/s.y());
            } else {
              iy=grid->indexOfY(s.y());
            }
            if(iy>-1 && iy<ny()) {
              //ASSERT(left(ix)<=s.x() && s.x()<=right(ix));
              //ASSERT(bottom(iy)<=s.y() && s.y()<=top(iy));
              (*grid->valuePointer(ix, iy))++;
            }
          }
        } else {
          stat++;
        }
      }
    }

    if(_ySampling & LogScale) {
      grid->exp(YAxis);
    } else if(_ySampling & InverseScale) {
      grid->inverse(YAxis);
    }
    App::log(1, tr("    %1/%2 samples are invalid\n").arg(stat).arg(_samples.count()));
  }

  VectorList<bool> Histogram2D::validSamplesFootprint(int ix) const
  {
    VectorList<bool> fp;
    for(int i=_samples.count()-1; i>=0; i--) {
      const Histogram2D::Sample& s=_samples.at(i);
      if(indexOfX(s.x())==ix) {
        fp.append(s.isValid());
      }
    }
    return fp;
  }

  int Histogram2D::validSampleCount(int ix) const
  {
    int n=0;
    for(int i=_samples.count()-1; i>=0; i--) {
      const Histogram2D::Sample& s=_samples.at(i);
      if(s.isValid() && indexOfX(s.x())==ix) {
        n++;
      }
    }
    return n;
  }

  void Histogram2D::setValid(const Rect& limits, bool v)
  {
    TRACE;
    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      if(limits.includes(s)) {
        s.setValid(v);
      }
    }
  }

  /*!
    The two histograms must have the same number of samples
    and their must have the same order. Copy the validity flags
    and the category from \a o into this.
  */
  void Histogram2D::setClassification(const Histogram2D * o)
  {
    ASSERT(_samples.count()==o->_samples.count());

    for(int i=_samples.count()-1; i>=0; i--) {
      Sample& s=_samples[i];
      s.setValid(o->_samples[i].isValid());
      s.setCategory(o->_samples[i].category());
    }
    countSamples();
  }

  Histogram Histogram2D::histogramAt(int index) const
  {
    TRACE;
    Histogram hist(stepCrossSection(YAxis, index));
    hist.setSampling(_ySampling);
    hist.setPeriodic(_periodic);
    hist.normalize();
    return hist;
  }

  VectorList<double> Histogram2D::sampleValues(int ix) const
  {
    TRACE;
    double min=left(ix);
    double max=right(ix);
    VectorList<double> v;

    if(_ySampling & LogScale) {
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples[i];
        if(s.isValid() && s.x()>min && s.x()<max) {
          v.append(::log(s.y()));
        }
      }
    } else if(_ySampling  & InverseScale) {
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples[i];
        if(s.isValid() && s.x()>min && s.x()<max) {
          v.append(1.0/s.y());
        }
      }
    } else {
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples[i];
        if(s.isValid() && s.x()>min && s.x()<max) {
          v.append(s.y());
        }
      }
    }
    return v;
  }

  /*!
    Returns mean computed on samples (not on classes).
  */
  RealStatisticalValue Histogram2D::mean(int ix) const
  {
    TRACE;
    double min=left(ix);
    double max=right(ix);
    if(_periodic) {
      PeriodicStatistics stat(y(ny()-1)-y(0));
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples[i];
        if(s.isValid() && s.x()>=min && s.x()<=max) {
          stat.meanAdd(s.y());
        }
      }
      stat.varianceReset();
      for(int i=_samples.count()-1; i>=0; i--) {
        const Sample& s=_samples[i];
        if(s.isValid() && s.x()>=min && s.x()<=max) {
          stat.varianceAdd(s.y());
        }
      }
      return RealStatisticalValue(stat.mean(), stat.stddev());
    } else {
      Statistics stat;
      if(_ySampling & LogScale) {
        for(int i=_samples.count()-1; i>=0; i--) {
          const Sample& s=_samples[i];
          if(s.isValid() && s.x()>min && s.x()<max) {
            stat.addLog(s.y());
          }
        }
        return RealStatisticalValue(stat.meanLog(), stat.stddevLog());
      } else if(_ySampling  & InverseScale) {
        for(int i=_samples.count()-1; i>=0; i--) {
          const Sample& s=_samples[i];
          if(s.isValid() && s.x()>min && s.x()<max) {
            stat.add(1.0/s.y());
          }
        }
        return RealStatisticalValue(1.0/stat.mean(), 1.0/stat.stddev());
      } else {
        for(int i=_samples.count()-1; i>=0; i--) {
          const Sample& s=_samples[i];
          if(s.isValid() && s.x()>min && s.x()<max) {
            stat.add(s.y());
          }
        }
        return RealStatisticalValue(stat.mean(), stat.stddev());
      }
    }
  }

  void Histogram2D::setStatistics(CurveProxy * curve, int index, double normFactor,
                                  const RealStatisticalValue& val) const
  {
    TRACE;
    curve->setY(index, val.mean(), nullptr);
    curve->setStddev(index, val.stddev());
    curve->setWeight(index, normFactor*val.weight());
    curve->setValid(index, true);
  }

  /*!
  */
  void Histogram2D::meanCurve(CurveProxy * curve) const
  {
    TRACE;
    Histogram2D hist(*this);
    countSamples(&hist);

    curve->resize(hist.nx());
    for(int i=0; i<hist.nx(); i++ ) {
      // hist is not normalized, values are number of samples
      // normFactor is used to get a valid weight
      double normFactor=hist.normalizeArea(YAxis, i);
      curve->setX(i, hist.x(i));
      if(normFactor>0.0) {
        setStatistics(curve, i, normFactor, hist.histogramAt(i).mean());
      } else {
        curve->setY(i, 0.0, nullptr);
        curve->setStddev(i, 0.0);
        curve->setValid(i, false);
        curve->setWeight(i, 0.0);
      }
    }
    curve->sort();
  }

  /*!
  */
  void Histogram2D::medianCurve(CurveProxy * curve) const
  {
    TRACE;
    Histogram2D hist(*this);
    countSamples(&hist);

    // Median computed on the real world values
    curve->resize(hist.nx());
    for(int i=0; i<hist.nx(); i++) {
      double normFactor=hist.normalizeArea(YAxis, i);
      curve->setX(i, hist.x(i));
      if(normFactor>0.0) {
        setStatistics(curve, i, normFactor, hist.histogramAt(i).median());
      } else {
        curve->setY(i, 0.0, nullptr);
        curve->setStddev(i, 0.0);
        curve->setValid(i, false);
        curve->setWeight(i, 0.0);
      }
    }
    curve->sort();
  }

  /*!
  */
  void Histogram2D::modeCurve(CurveProxy * curve) const
  {
    TRACE;
    Histogram2D hist(*this);
    countSamples(&hist);

    curve->resize(hist.nx());
    for(int i=0; i<hist.nx(); i++) {
      double normFactor=hist.normalizeArea(YAxis, i);
      curve->setX(i, hist.x(i));
      if(normFactor>0.0) {
        setStatistics(curve, i, normFactor, hist.histogramAt(i).mode());
      } else {
        curve->setY(i, 0.0, nullptr);
        curve->setStddev(i, 0.0);
        curve->setValid(i, false);
        curve->setWeight(i, 0.0);
      }
    }
    curve->sort();
  }

  /*!
  */
  void Histogram2D::removeOutliers(int ix, double factor)
  {
    TRACE;
    VectorList<double> values1=sampleValues(ix);
    VectorList<double> values2;
    double sum=0.0, sum2=0.0;
    for(int i=values1.count()-1; i>=0; i--) {
      double v=values1.at(i);
      sum+=v;
      sum2+=v*v;
    }
    values2.reserve(values1.count());

    VectorList<double> * originalValues=&values1;
    VectorList<double> * filteredValues=&values2;
    for(int ir=0; ir<10; ir++) {
      int n=originalValues->count();
      if(n>2) {
        double mean=sum/n;
        double threshold=factor*::sqrt((sum2-mean*sum)/(n-1));
        for(int i=n-1; i>=0; i--) {
          double v=originalValues->at(i);
          if(fabs(v-mean)>threshold) {
            sum-=v;
            sum2-=v*v;
          } else {
            filteredValues->append(v);
          }
        }
        if(originalValues->count()==filteredValues->count()) {
          break;
        }
        qSwap(originalValues, filteredValues);
      }
    }
  }

#define CLASSIFY_ACTION0(condition) \
  switch(param.action()) { \
  case SampleClassificationParameters::Invalidate: \
    condition<Invalidate>(param); \
    break; \
  case SampleClassificationParameters::Validate: \
    condition<Validate>(param); \
    break; \
  case SampleClassificationParameters::Categorize: \
    condition<Categorize>(param); \
    break; \
  case SampleClassificationParameters::Reset: \
    condition<Reset>(param); \
    break; \
  }

#define CLASSIFY_ACTION1(condition, arg1) \
  switch(param.action()) { \
  case SampleClassificationParameters::Invalidate: \
    condition<Invalidate>(arg1, param); \
    break; \
  case SampleClassificationParameters::Validate: \
    condition<Validate>(arg1, param); \
    break; \
  case SampleClassificationParameters::Categorize: \
    condition<Categorize>(arg1, param); \
    break; \
  case SampleClassificationParameters::Reset: \
    condition<Reset>(arg1, param); \
    break; \
  }

#define CLASSIFY_ACTION2(condition, arg1, arg2) \
  switch(param.action()) { \
  case SampleClassificationParameters::Invalidate: \
    condition<Invalidate>(arg1, arg2, param); \
    break; \
  case SampleClassificationParameters::Validate: \
    condition<Validate>(arg1, arg2, param); \
    break; \
  case SampleClassificationParameters::Categorize: \
    condition<Categorize>(arg1, arg2, param); \
    break; \
  case SampleClassificationParameters::Reset: \
    condition<Reset>(arg1, arg2, param); \
    break; \
  }

  void Histogram2D::classifySamples(const SampleClassificationParameters& param,
                                    CurveProxy * curve1,
                                    CurveProxy * curve2)
  {
    TRACE;
    switch(param.valueConditionType()) {
    case SampleClassificationParameters::NoCondition:
      CLASSIFY_ACTION0(all)
      break;
    case SampleClassificationParameters::AbsoluteThreshold:
      CLASSIFY_ACTION0(threshold)
      break;
    case SampleClassificationParameters::RelativeThreshold:
      normalizeAmplitude(YAxis);
      CLASSIFY_ACTION0(threshold)
      break;
    case SampleClassificationParameters::CurveAbsoluteInsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveInsideAbsoluteRange, curve1)
      break;
    case SampleClassificationParameters::CurveRelativeInsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveInsideRelativeRange, curve1)
      break;
    case SampleClassificationParameters::CurveStddevInsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveInsideSigmaRange, curve1)
      break;
    case SampleClassificationParameters::CurveAbsoluteOutsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveOutsideAbsoluteRange, curve1)
      break;
    case SampleClassificationParameters::CurveRelativeOutsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveOutsideRelativeRange, curve1)
      break;
    case SampleClassificationParameters::CurveStddevOutsideRange:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(curveOutsideSigmaRange, curve1)
      break;
    case SampleClassificationParameters::AboveCurve:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(aboveCurve, curve1)
      break;
    case SampleClassificationParameters::BelowCurve:
      ASSERT(curve1);
      curve1->sort();
      CLASSIFY_ACTION1(belowCurve, curve1)
      break;
    case SampleClassificationParameters::InsideCurves:
      ASSERT(curve1);
      ASSERT(curve2);
      curve1->sort();
      curve2->sort();
      CLASSIFY_ACTION2(insideCurves, curve1, curve2)
      break;
    case SampleClassificationParameters::OutsideCurves:
      ASSERT(curve1);
      ASSERT(curve2);
      curve1->sort();
      curve2->sort();
      CLASSIFY_ACTION2(outsideCurves, curve1, curve2)
      break;
    }
  }

  /*!
    Interpoles the mean at \a x for \a curve.
    \a arg is an absolute deviation from the mean.
    The range is returned in \a ymin and \a ymax.
  */
  void Histogram2D::admissibleAbsoluteRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax)
  {
    double mean, stddev;
    curve->interpole(x, mean, stddev, nullptr, _xSampling, _ySampling);
    double dy;
    switch(_yScaleType) {
    case Scale::InverseLog:
    case Scale::Log:
      ymin=mean/arg;
      ymax=mean*arg;
      break;
    case Scale::Inverse:
      dy=1.0/mean;
      ymin=1.0/(dy+arg);
      ymax=1.0/(dy-arg);
      break;
    case Scale::Linear:
    case Scale::AbsoluteTime:
    case Scale::RelativeTime:
      ymin=mean-arg;
      ymax=mean+arg;
      break;
    }

  }

  /*!
    Interpoles the mean at \a x for \a curve.
    \a arg is a factor (for linear scale) for the mean.
    The range is returned in \a ymin and \a ymax.
  */
  void Histogram2D::admissibleRelativeRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax)
  {
    double mean, stddev;
    curve->interpole(x, mean, stddev, nullptr, _xSampling, _ySampling);
    double dy;
    switch(_yScaleType) {
    case Scale::InverseLog:
    case Scale::Log:
      dy=1.0+arg;
      ymin=mean/dy;
      ymax=mean*dy;
      break;
    case Scale::Inverse:
      ymin=mean/(1.0+arg);
      ymax=mean/(1.0-arg);
      break;
    case Scale::Linear:
    case Scale::AbsoluteTime:
    case Scale::RelativeTime:
      dy=mean*arg;
      ymin=mean-dy;
      ymax=mean+dy;
      break;
    }

  }

  /*!
    Interpoles the mean and the standard deviation at \a x for \a curve.
    \a arg is a factor (for linear scale) for the standard deviation.
    The range is returned in \a ymin and \a ymax.
  */
  void Histogram2D::admissibleSigmaRange(double x, const CurveProxy * curve, double arg, double& ymin, double& ymax)
  {
    double mean, stddev;
    curve->interpole(x, mean, stddev, nullptr, _xSampling, _ySampling);
    double dy;
    switch(_yScaleType) {
    case Scale::InverseLog:
    case Scale::Log:
      dy=pow(stddev, arg);
      ymin=mean/dy;
      ymax=mean*dy;
      break;
    case Scale::Inverse:
      dy=stddev/mean;
      ymin=stddev/(dy+arg);
      ymax=stddev/(dy-arg);
      break;
    case Scale::Linear:
    case Scale::AbsoluteTime:
    case Scale::RelativeTime:
      dy=stddev*arg;
      ymin=mean-dy;
      ymax=mean+dy;
      break;
    }

  }


  /*!

  */
  void Histogram2D::filterMisfit(const CurveProxy * curve, double maxMisfit)
  {
    TRACE;
    IrregularGrid2D grid=*this;
    if(_ySampling & LogScale) {
      grid.log(YAxis);
    } else if(_ySampling & InverseScale) {
      grid.inverse(YAxis);
    }

    int nx=curve->sampleCount();
    double mean, stddev;
    for(int ix=0; ix<nx; ix++) {
      if(curve->isValid(ix)) {
        mean=curve->y(ix, nullptr);
        stddev=curve->stddev(ix);
        NormalDistribution gd(mean, stddev);
        misfit(curve->x(ix), grid, gd, maxMisfit);
      }
    }
    countSamples();
  }

  bool Histogram2D::save(const QString &fileName, const QString& headerLine, bool maxFormat) const
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      if(!headerLine.isEmpty()) {
        if(maxFormat) {
          s << "# MAX FORMAT RELEASE 1.1\n";
          CoreApplication::signature(s);
          s << "#\n"
               "# BEGIN DATA\n"
               "#\n";
        }
        s << headerLine;
      }
      write(s);
      return true;
    } else {
      return false;
    }
  }

  bool Histogram2D::load(const QString &fileName)
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      QTextStream s(&f);
      read(s);
      return true;
    } else {
      return false;
    }
  }

  void Histogram2D::write(QTextStream& s) const
  {
    TRACE;
    VectorList<Sample>::const_iterator it;
    if(!_samples.isEmpty()) {
      if(_samples.first().line().isEmpty()) {
        for(it=_samples.begin(); it!=_samples.end(); ++it) {
          it->writeData(s);
        }
      } else {
        for(it=_samples.begin(); it!=_samples.end(); ++it) {
          it->writeLine(s);
        }
      }
    }
  }

  void Histogram2D::read(QTextStream& s)
  {
    TRACE;
    Sample sample;
    QString line;
    while(true) {
      if(File::readLine(s, line) && sample.read(line)) {
        _samples.append(sample);
      } else {
        break;
      }
    }
    countSamples();
  }

  /*!
    Compute misfit between theoretical normal distribution based on \a curve and the histogram
    If misfit is greater than \a maxMisfit, the sample of \a curve is set invalid.
    Cumulative function is compared and not the density function.
  */
  void Histogram2D::validateCurve(Curve<RealStatisticalPoint>& curve, double maxMisfit) const
  {
    TRACE;
    IrregularGrid2D grid=*this;
    if(_ySampling & LogScale) {
      grid.log(YAxis);
    } else if(_ySampling & InverseScale) {
      grid.inverse(YAxis);
    }

    int nx=curve.count();
    for(int ix=0; ix<nx; ix++) {
      RealStatisticalPoint& sc=curve.constXAt(ix);
      if(sc.isValid()) {
        NormalDistribution gd(sc.mean(), sc.stddev());
        if(misfit(sc.x(), grid, gd)>maxMisfit) {
          sc.setValid(false);
        }
      }
    }
  }

  /*!
    Compute misfit between theoretical normal distribution and a cross-section at \a x.
    Cumulative function is compared and not the density function.

    Return -1 is histogram is empty.
  */
  double Histogram2D::misfit(double x, const NormalDistribution& gd) const
  {
    TRACE;
    IrregularGrid2D grid=*this;
    if(_ySampling & LogScale) {
      grid.log(YAxis);
    } else if(_ySampling & InverseScale) {
      grid.inverse(YAxis);
    }
    return misfit(x, grid, gd);
  }

  /*!
    Compute misfit between theoretical normal distribution and a cross-section at \a x.
    Cumulative function is compared and not the density function.

    Return -1 is histogram is empty.
  */
  double Histogram2D::misfit(double x, IrregularGrid2D& grid, const NormalDistribution& gd) const
  {
    TRACE;
    int ix=grid.indexOfX(x);
    if(ix<0) {
      ix++;
    } else if(ix==grid.nx()) {
      ix--;
    }
    return misfit(ix, grid, gd);
  }

  /*!
    Compute misfit between theoretical normal distribution and a cross-section at index \a ix.
    Cumulative function is compared and not the density function.

    Return -1 if histogram is empty.
  */
  double Histogram2D::misfit(int ix, IrregularGrid2D& grid, const NormalDistribution& gd) const
  {
    TRACE;
    double factor=grid.normalizeArea(YAxis, ix);
    if(factor==0.0) {
      return -1.0;
    }
    Curve<Point2D> c=grid.crossSection(YAxis, ix);
    if(_ySampling & LogScale) {
      c.xExp();
    } else if(_ySampling & InverseScale) {
      c.xInverse();
    }

    int ny=c.count();
    double m=0.0, sum=0.0;
    for(int iy=0; iy<ny; iy++) {
      const Point2D& p=c.constAt(iy);
      sum+=p.y()*grid.height(iy);
      double ygd;
      if(_ySampling & LogScale) {
        ygd=gd.cumulativeValue(::log(p.x()));
      } else if(_ySampling & InverseScale) {
        ygd=gd.cumulativeValue(1.0/p.x());
      } else {
        ygd=gd.cumulativeValue(p.x());
      }
      double dy=sum-ygd;
      m+=dy*dy;
    }
    m/=ny;
    return ::sqrt(m);
  }

  /*!
    Compute misfit between theoretical normal distribution and a cross-section at \a x.
    Cumulative function is compared and not the density function.

    Samples are set invalid where local misfit is less than \a maxMisfit.
    Filter out samples that fit \a gd at \a x.
  */
  void Histogram2D::misfit(double x, IrregularGrid2D& grid, const NormalDistribution& gd, double maxMisfit)
  {
    TRACE;
    int ix=grid.indexOfX(x);
    if(ix<0) {
      ix++;
    } else if(ix==grid.nx()) {
      ix--;
    }
    double factor=grid.normalizeArea(YAxis, ix);
    if(factor==0.0) {
      return;
    }
    Curve<Point2D> c=grid.crossSection(YAxis, ix);
    if(_ySampling & LogScale) {
      c.xExp();
    } else if(_ySampling & InverseScale) {
      c.xInverse();
    }

    int ny=c.count();
    double sum=0.0;
    for(int iy=0; iy<ny; iy++) {
      const Point2D& p=c.constAt(iy);
      sum+=p.y()*grid.height(iy);
      double ygd;
      if(_ySampling & LogScale) {
        ygd=gd.cumulativeValue(::log10(p.x()));
      } else if(_ySampling & InverseScale) {
        ygd=gd.cumulativeValue(1.0/p.x());
      } else {
        ygd=gd.cumulativeValue(p.x());
      }
      double dy=sum-ygd;
      if(::fabs(dy)<maxMisfit) {
        VectorList<Sample>::iterator it;
        for(it=_samples.begin(); it!=_samples.end(); ++it) {
          Sample& s=*it;
          if(s.isValid() && grid.belongs(s, ix, iy)) {
            qDebug() << s.x() << s.y() << p.x() << sum << ygd << ::fabs(dy) << std::abs(dy);
            s.setValid(false);
          }
        }
      }
    }
  }

  bool saveCurve(const Curve<RealStatisticalPoint>& c, const QString &fileName)
  {
    TRACE;
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      s.setRealNumberPrecision(20);
      int n=c.count();
      for(int i=0; i<n; i++) {
        s << c.constAt(i).toString('g', 20) << "\n";
      }
      return true;
    } else {
      return false;
    }
  }

  /*!
    Automatically pick all maxima

    \a maxCount maxima per x. \a range is the relative tolerance (0.1 for a 10% range)
    around the mean.
  */
  Curve<RealStatisticalPoint> Histogram2D::pickAll(int maxCount, double exploreRange, double maxError) const
  {
    Curve<RealStatisticalPoint> curve;
    RealStatisticalPoint p;
    for(int ix=0; ix<nx(); ix++) {
      p.setX(x().at(ix));
      Histogram h=histogramAt(ix);
      VectorList<RealStatisticalValue> values=h.pick(maxCount, exploreRange, maxError);
      for(int i=0; i<values.count(); i++) {
        p=values.at(i);
        curve.append(p);
      }
    }
    return curve;
  }

  /*!
    Automatically pick all maxima with a Gaussian fit.
  */
  Curve<RealStatisticalPoint> Histogram2D::pickAll(double maxMisfit) const
  {
    int i=1;
    Histogram2D h=*this;
    Curve<RealStatisticalPoint> all;
    RealStatisticalProxy proxy;
    //c=h.modeFitCurve();
    saveCurve(proxy.curve(), QString("/tmp/curve_%1.txt").arg(i, 4, 10, QChar('0')));
    h.save(QString("/tmp/histo_%1.txt").arg(i, 4, 10, QChar('0')));
    while(!proxy.curve().isEmpty()) {
      h.filterMisfit(&proxy, maxMisfit);
      all.append(proxy.curve());
      //c=h.modeFitCurve();
      i++;
      saveCurve(proxy.curve(), QString("/tmp/curve_%1.txt").arg(i, 4, 10, QChar('0')));
      h.save(QString("/tmp/histo_%1.txt").arg(i, 4, 10, QChar('0')));
    }
    return all;
  }

  SamplingParameters Histogram2D::xSamplingParameters() const
  {
    SamplingParameters p;
    p.setScaleType(SamplingParameters::scaleType(xSampling()));
    p.setCount(nx());
    p.setRange(minimumAxis(XAxis), maximumAxis(XAxis));
    return p;
  }

  SamplingParameters Histogram2D::ySamplingParameters() const
  {
    SamplingParameters p;
    p.setScaleType(SamplingParameters::scaleType(ySampling()));
    p.setCount(ny());
    p.setRange(minimumAxis(YAxis), maximumAxis(YAxis));
    return p;
  }

} // namespace QGpCoreStat
