/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-09-24
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef VECTOR_H
#define VECTOR_H

#include "QGpCoreMathDLLExport.h"
#include "Matrix.h"

namespace QGpCoreMath {

  template <typename T> class QGPCOREMATH_EXPORT Vector
  {
  public:
    inline Vector(int count, T * values);
    inline Vector(const Vector<T>& o);
    inline ~Vector() {}

    void operator=(const Vector<T>&) {ASSERT(false);}
    void copyValues(const Vector<T>& o);

    inline void operator+=(const Vector<T>& o);
    inline void operator-=(const Vector<T>& o);
    inline void operator*=(const T& value);
    inline void operator/=(const T& value);

    inline bool operator==(const Vector<T>& o) const;
    inline bool operator!=(const Vector<T>& o) const;
    inline bool operator<(const Vector<T>& o) const;
    inline bool operator<=(const Vector<T>& o) const;
    inline bool operator>(const Vector<T>& o) const;
    inline bool operator>=(const Vector<T>& o) const;
    inline bool hasElementGreaterThan(const Vector<T>& o) const;

    int count() const {return _count;}

    T& at(int index) {return _values[index];}
    const T& at(int index) const {return _values[index];}

    T& operator[](int index) {return _values[index];}
    const T& operator[](int index) const {return _values[index];}

    inline void setValue(const T& value);
    void setValues(T * values) {_values=values;}
    void setValues(const Vector<T>& o) {_values=o._values;}
    T * values() {return _values;}
    const T * values() const {return _values;}

    inline Vector<T>& init(const T& v1);
    inline Vector<T>& init(const T& v1, const T& v2);
    inline Vector<T>& init(const T& v1, const T& v2, const T& v3);
    inline Vector<T>& init(const T& v1, const T& v2, const T& v3, const T& v4);

    double distanceTo(const Vector<T>& o) const;
    double distanceTo(const Vector<T>& o, int axis1, int axis2) const;
    inline double length() const;
    inline double squareLength() const;
    inline double length(int axis1, int axis2) const;
    inline double squareLength(int axis1, int axis2) const;
    inline double azimuth(int axis1, int axis2) const;

    double minimumAbsRatio(const Vector<T>& o) const;

    template <typename T1, typename T2>
    inline void multiply(const Matrix<T1>& m, const Vector<T2>& v);
    template <typename T1, typename T2>
    inline void multiply(const Vector<T1>& v, const Matrix<T2>& m);
    template <typename T1, typename T2>
    inline void dotMultiply(const Vector<T1>& v1, const Vector<T2>& v2);
    inline Matrix<T> crossMatrix() const;
    template <typename T1> inline T scalarProduct(const Vector<T1>& o) const;
    inline T scalarProduct(const Point2D& p) const;
    template <typename T1>
    inline T scalarProductConjugate(const Vector<T1>& o) const;
    inline void abs();
    inline void changeSign();

    QString toString(char format='g', int precision=6) const;
    QString toUserString(char format='g', int precision=6) const;
    QString toOctaveString(char format='g', int precision=6) const;
    void toOctaveFile(const QString& fileName, const QString& variableName) const;
  protected:
    void resize(int count) {_count=count;}
  private:
    int _count;
    T * _values;
  };

  QGPCOREMATH_EXPORT VectorList<double> qVector(const Vector<double>& v);

  template <typename T>
  inline Vector<T>::Vector(int count, T * values)
  {
    _count=count;
    _values=values;
  }

  template <typename T>
  inline Vector<T>::Vector(const Vector<T>& o)
  {
    _count=o._count;
    _values=o._values;
  }

  template <typename T>
  inline void Vector<T>::copyValues(const Vector<T>& o)
  {
    ASSERT(_count==o._count);
    memcpy(_values, o._values, _count*sizeof(T));
  }

  template <typename T>
  inline void Vector<T>::operator+=(const Vector<T>& o)
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      _values[i]+=o._values[i];
    }
  }

  template <typename T>
  inline void Vector<T>::operator-=(const Vector<T>& o)
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      _values[i]-=o._values[i];
    }
  }

  template <typename T>
  inline void Vector<T>::operator*=(const T& value)
  {
    for(int i=0; i<_count; i++) {
      _values[i]*=value;
    }
  }

  template <typename T>
  inline void Vector<T>::operator/=(const T& value)
  {
    for(int i=0; i<_count; i++) {
      _values[i]/=value;
    }
  }

  template <typename T>
  inline bool Vector<T>::operator==(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]!=o._values[i]) {
        return false;
      }
    }
    return true;
  }

  template <typename T>
  inline bool Vector<T>::operator!=(const Vector<T>& o) const
  {
    for(int i=0; i<_count; i++) {
      if(_values[i]!=o._values[i]) {
        return true;
      }
    }
    return false;
  }

  template <typename T>
  inline bool Vector<T>::operator<(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]<o._values[i]) {
        return true;
      } else if(_values[i]>o._values[i]) {
        return false;
      }
    }
    // All elements are equal
    return false;
  }

  template <typename T>
  inline bool Vector<T>::operator<=(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]<o._values[i]) {
        return true;
      } else if(_values[i]>o._values[i]) {
        return false;
      }
    }
    // All elements are equal
    return true;
  }

  template <typename T>
  inline bool Vector<T>::operator>(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]>o._values[i]) {
        return true;
      } else if(_values[i]<o._values[i]) {
        return false;
      }
    }
    // All elements are equal
    return false;
  }

  template <typename T>
  inline bool Vector<T>::operator>=(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]>o._values[i]) {
        return true;
      } else if(_values[i]<o._values[i]) {
        return false;
      }
    }
    // All elements are equal
    return true;
  }

  template <typename T>
  inline bool Vector<T>::hasElementGreaterThan(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    for(int i=0; i<_count; i++) {
      if(_values[i]>o._values[i]) {
        return true;
      }
    }
    return false;
  }

  template <typename T>
  inline void Vector<T>::setValue(const T& value)
  {
    for(int i=0; i<_count; i++) {
      _values[i]=value;
    }
  }

  template <typename T>
  double Vector<T>::distanceTo(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    double sum=0.0, x;
    for(int i=0; i<_count; i++) {
      x=_values[i]-o._values[i];
      sum+=x*x;
    }
    return sum;
  }

  template <typename T>
  double Vector<T>::distanceTo(const Vector<T>& o, int axis1, int axis2) const
  {
    ASSERT(axis2<_count && axis2<o._count && axis1<axis2);
    double sum=0.0, x;
    x=_values[axis1];
    x-=o._values[axis1];
    sum+=x*x;
    x=_values[axis2];
    x-=o._values[axis2];
    sum+=x*x;
    return ::sqrt(sum);
  }

  template <typename T>
  inline double Vector<T>::squareLength() const
  {
    double sum=0.0, x;
    for(int i=0; i<_count; i++) {
      x=_values[i];
      sum+=x*x;
    }
    return sum;
  }

  template <typename T>
  inline double Vector<T>::length() const
  {
    return ::sqrt(squareLength());
  }

  template <typename T>
  inline double Vector<T>::squareLength(int axis1, int axis2) const
  {
    ASSERT(axis1<_count && axis2<_count);
    double sum=0.0, x;
    x=_values[axis1];
    sum+=x*x;
    x=_values[axis2];
    sum+=x*x;
    return sum;
  }

  template <typename T>
  inline double Vector<T>::length(int axis1, int axis2) const
  {
    return ::sqrt(squareLength(axis1, axis2));
  }

  template <typename T>
  inline double Vector<T>::azimuth(int axis1, int axis2) const
  {
    ASSERT(axis1<_count && axis2<_count);
    double a=atan2(_values[axis2], _values[axis1]);
    if(a<0.0) a+=2.0*M_PI;
    return a;
  }

  template <typename T>
  inline Vector<T>& Vector<T>::init(const T& v1)
  {
    ASSERT(_count==1);
    _values[0]=v1;
    return *this;
  }

  template <typename T>
  inline Vector<T>& Vector<T>::init(const T& v1, const T& v2)
  {
    ASSERT(_count==2);
    _values[0]=v1;
    _values[1]=v2;
    return *this;
  }

  template <typename T>
  inline Vector<T>& Vector<T>::init(const T& v1, const T& v2, const T& v3)
  {
    ASSERT(_count==3);
    _values[0]=v1;
    _values[1]=v2;
    _values[2]=v3;
    return *this;
  }

  template <typename T>
  inline Vector<T>& Vector<T>::init(const T& v1, const T& v2, const T& v3, const T& v4)
  {
    ASSERT(_count==4);
    _values[0]=v1;
    _values[1]=v2;
    _values[2]=v3;
    _values[3]=v4;
    return *this;
  }

  template <typename T>
  template <typename T1, typename T2>
  inline void Vector<T>::multiply(const Matrix<T1>& m, const Vector<T2>& v)
  {
    ASSERT(static_cast<int>(m.columnCount())==v.count());
    ASSERT(_count==v.count());
    setValue(0);
    T * r=_values;
    const T * m1=m.values();
    const T * m2=v._values;
    const T * m1_end=m1+m.rowCount();
    const T * m10, * m20, * m20_end;
    while(m1<m1_end) {
      m10=m1;
      m20=m2;
      m20_end=m2+m.columnCount();
      while(m20<m20_end) {
        *r+=(*m10)*(*m20);
        m10+=m.rowCount();
        m20++;
      }
      m1++;
      r++;
    }
  }

  template <typename T>
  template <typename T1, typename T2>
  inline void Vector<T>::multiply(const Vector<T1>& v, const Matrix<T2>& m)
  {
    ASSERT(m.count()==v.count());
    ASSERT(_count==v._count);
    setValue(0);
    T * r=_values;
    const T * m1=v._values;
    const T * m2=m.values();
    const T * m2_end=m2+m.columnCount()*m.rowCount();
    const T * m10, * m20, * m10_end;
    T x;
    while(m2<m2_end) {
      m10=m1;
      m20=m2;
      m10_end=m1+v._count;
      while(m10<m10_end) {
        x=*m10;
        x*=*m20
        *r+=x;
        m20++;
        m10++;
      }
      m2+=m.rowCount();
      r++;
    }
  }

  template <typename T>
  template <typename T1, typename T2>
  inline void Vector<T>::dotMultiply(const Vector<T1>& v1, const Vector<T2>& v2)
  {
    ASSERT(_count==v1.count() && v1.count()==v2.count());
    for(int i=0; i<_count; i++) {
      T& val=_values[i];
      val=v1.at(i);
      val*=v2.at(i);
    }
  }

  template <typename T>
  template <typename T1>
  inline T Vector<T>::scalarProduct(const Vector<T1>& o) const
  {
    ASSERT(_count==o.count());
    T sum=0, x;
    for(int i=0; i<_count; i++) {
      x=_values[i];
      x*=o._values[i];
      sum+=x;
    }
    return sum;
  }

  template <typename T>
  inline Matrix<T> Vector<T>::crossMatrix() const
  {
    Matrix<T> m(_count);
    for(int i=0; i<_count; i++) {
      for(int j=0; j<_count; j++) {
        m.at(i, j)=_values[i]*conjugate(_values[j]);
      }
    }
    return m;
  }

  template <typename T>
  template <typename T1>
  inline T Vector<T>::scalarProductConjugate(const Vector<T1>& o) const
  {
    ASSERT(_count==o.count());
    T sum=0, x;
    for(int i=0; i<_count; i++) {
      x=conjugate(o._values[i]);
      x*=_values[i];
      sum+=x;
    }
    return sum;
  }

  template <typename T>
  inline T Vector<T>::scalarProduct(const Point2D& p) const
  {
    ASSERT(_count>=2);
    T x=_values[0];
    x*=p.x();
    T y=_values[1];
    y*=p.y();
    x+=y;
    return x;
  }

  template <typename T>
  inline void Vector<T>::abs()
  {
    for(int i=0; i<_count; i++) {
      _values[i]=QGpCoreTools::abs(_values[i]);
    }
  }

  template <typename T>
  inline void Vector<T>::changeSign()
  {
    for(int i=0; i<_count; i++) {
      _values[i]=-_values[i];
    }
  }

  template <typename T>
  QString Vector<T>::toString(char format, int precision) const
  {
    QString str;
    for(int i=0; i<_count; i++) {
      str+=Number::toString(_values[i], format, precision);
      str+=" ";
    }
    return str;
  }

  template <typename T>
  QString Vector<T>::toUserString(char format, int precision) const
  {
    QString str;
    if(_count>0) {
      str=QString("%1").arg(_values[0], 12, format, precision, QChar(' '));
      for(int i=1; i<_count; i++) {
        str+=QString(" %1").arg(_values[i], 12, format, precision, QChar(' '));
      }
    }
    return str;
  }

  template <typename T>
  QString Vector<T>::toOctaveString(char format, int precision) const
  {
    QString str("[");
    for(int i=0; i<_count; i++) {
      str+=QString(" %1;").arg(Number::toString(_values[i], format, precision));
    }
    str+=" ]";
    return str;
  }

  template <typename T> void Vector<T>::toOctaveFile(const QString& fileName, const QString& variableName) const
  {
    QFile f(fileName);
    if(!f.open(QIODevice::Append)) {
      App::log(tr("Cannot write to %1\n").arg(fileName));
      return;
    }
    QTextStream s(&f);
    s << variableName << "=" << toOctaveString('g', 20) << ";" << Qt::endl;
    f.close();
  }

  template <typename T>
  double Vector<T>::minimumAbsRatio(const Vector<T>& o) const
  {
    ASSERT(_count==o._count);
    double minR=std::numeric_limits<double>::infinity();
    double r;
    for(int i=0; i<_count; i++) {
      r=fabs(_values[i]/o._values[i]);
      if(r<minR) {
        minR=r;
      }
    }
    return minR;
  }


} // namespace QGpCoreMath

#endif // VECTOR_H

