/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-11-09
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef POINTND_H
#define POINTND_H

#include <QGpCoreTools.h>

#include "SamplingParameters.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

class CurvePointOptions;
class Point2D;

class QGPCOREMATH_EXPORT PointND
{
public:
  PointND() {_x=0.0;}
  PointND(double x, const VectorList<double>& y) {_x=x; _y=y;}
  PointND(const PointND& o) {_x=o._x; _y=o._y;}
  PointND(const VectorList<double>& p);

  int count() const {return _y.count();}

  bool operator<(const PointND& p) const {return _x<p._x;}
  inline bool operator==(const PointND& p) const;
  inline void operator=(const PointND& p);

  double x() const {return _x;}
  void setX(double x) {_x=x;}
  double y(const CurvePointOptions * options) const;
  void setY(double y, const CurvePointOptions * options);
  Point2D point2D(const CurvePointOptions * options) const;
  inline double y(int index) const;
  inline void setY(int index, double y);
  double at(int index) const;

  void setValid(bool) {}
  bool isValid() const {return true;}
  inline void interpole(double valX, const PointND& p1, const PointND& p2,
                        SamplingOptions xSampling, SamplingOptions ySampling);
  inline void average(const PointND&);

  void operator+=(const PointND& p);
  void operator*=(double mul);
  void operator/=(double div) {operator*=(1.0/div);}
  void yLog();
  void yExp();
  void yInverse();

  // I/O functions
  bool fromString(const StringSection& str);
  QString toString(char format='g', int precision=6) const;
private:
  double _x;
  VectorList<double> _y;
};

inline bool PointND::operator==(const PointND& p) const
{
  if(_x!=p._x) return false;
  int n=_y.count();
  if(n!=p._y.count()) return false;
  for(int i=0; i< n; i++) {
    if(_y.at(i)!=p._y.at(i)) return false;
  }
  return true;
}

inline void PointND::operator=(const PointND& p)
{
  _x=p._x;
  _y=p._y;
}

inline double PointND::y(int index) const
{
  TRACE;
  if(index >= _y.count()) {
    if(_y.isEmpty()) return 0.0; else return _y.last();
  } else {
    return _y[index];
  }
}

inline void PointND::setY(int index, double y)
{
  TRACE;
  if(index>=_y.count()) {
    _y.resize(index+1);
  }
  _y[index]=y;
}

inline void PointND::interpole(double valX, const PointND& p1, const PointND& p2,
                               SamplingOptions xSampling, SamplingOptions ySampling)
{
  TRACE;
  setX(valX);
  double x1, x2;
  double y1, y2;
  if(xSampling & InverseScale) {
    valX=1.0/valX;
    x1=1.0/p1.x();
    x2=1.0/p2.x();
  } else if(xSampling & LogScale) {
    valX=log(valX);
    x1=log(p1.x());
    x2=log(p2.x());
  } else {
    x1=p1.x();
    x2=p2.x();
  }
  double factor=(valX-x1)/(x2-x1);
  int n=p1.count();
  if(n<p2.count()) {
    n=p2.count();
  }
  if(ySampling & InverseScale) {
    for(int i=0; i< n; i++) {
      y1=1.0/p1.y(i);
      y2=1.0/p2.y(i);
      this->setY(i, 1.0/(y1+(y2-y1)*factor));
    }
  } else if(ySampling & LogScale) {
    for(int i=0; i< n; i++) {
      y1=log(p1.y(i));
      y2=log(p2.y(i));
      this->setY(i, exp(y1+(y2-y1)*factor));
    }
  } else {
    for(int i=0; i< n; i++) {
      y1=p1.y(i);
      y2=p2.y(i);
      this->setY(i, y1+(y2-y1)*factor);
    }
  }
}

inline void PointND::average(const PointND& p)
{
  TRACE;
  int n=p.count();
  if(n<count()) n=count();
  for(int i=0; i< n; i++) {
    setY(i, 0.5*(y(i) + p.y(i)));
  }
}

} // namespace QGpCoreMath

#endif // POINTND_H
