/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-01-16
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef PERIODICSTATISTICS_H
#define PERIODICSTATISTICS_H

#include <QtCore>

#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT PeriodicStatistics
  {
  public:
    PeriodicStatistics(double period);

    void meanAdd(double val, double weight=1.0);
    void setAverage(double average) {_currentAverage=average;}
    void varianceAdd(double val, double weight=1.0);
    void meanReset();
    void varianceReset();

    int count() const {return qRound(_sumWeight);}
    double mean() const;
    double variance() const;
    double stddev() const;
    double relativeStddev() const {return stddev()/_currentAverage;}
    double weight() const {return _sumWeight;}

    static void test();
    static inline void shiftFunction(double& x, const double& mean,
                                     const double& period, const double& halfPeriod);
    static inline void shiftValue(double& v,
                                  const double& period, const double& halfPeriod);
    static inline void shiftValue(double& v, const double& vRef,
                                  const double& period, const double& halfPeriod);
  private:
    double _period, _halfPeriod, _doublePeriod;
    double _sum, _sum2, _sumWeight, _sumWeight2;
    double _currentAverage;
  };

  inline void PeriodicStatistics::shiftFunction(double& x, const double& mean,
                                                const double& period, const double& halfPeriod)
  {
    double diff=x-mean;
    if(diff<-halfPeriod) {
      x+=period;
    } else if(diff>halfPeriod) {
      x-=period;
    }
  }

  inline void PeriodicStatistics::shiftValue(double& v,
                                             const double& period, const double& halfPeriod)
  {
    if(v<-halfPeriod) {
      v+=period;
    } else if(v>halfPeriod) {
      v-=period;
    }
  }

  inline void PeriodicStatistics::shiftValue(double& v, const double& vRef,
                                             const double& period, const double& halfPeriod)
  {
    double diff=v-vRef;
    if(diff<-halfPeriod) {
      v+=period;
    } else if(diff>halfPeriod) {
      v-=period;
    }
  }

} // namespace QGpCoreMath

#endif // PERIODICSTATISTICS_H

