/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-01-16
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PeriodicStatistics.h"
#include "Random.h"
#include "Statistics.h"

namespace QGpCoreMath {

  /*!
    \class PeriodicStatistics PeriodicStatistics.h
    \brief Statistics of a periodic variable

    Continuously check that values are in the same cycle
    before summing them.
  */

  /*!
    Description of constructor still missing
  */
  PeriodicStatistics::PeriodicStatistics(double period)
  {
    _period=period;
    _halfPeriod=0.5*_period;
    _doublePeriod=2.0*_period;
    meanReset();
    varianceReset();
  }

  void PeriodicStatistics::meanReset()
  {
    _sum=0.0;
    _sumWeight=0.0;
    _currentAverage=std::numeric_limits<double>::quiet_NaN();
  }

  void PeriodicStatistics::varianceReset()
  {
    _sum=0.0;
    _sum2=0.0;
    _sumWeight=0.0;
    _sumWeight2=0.0;
  }

  /*!
    Add samples to compute the mean.
  */
  void PeriodicStatistics::meanAdd(double val, double weight)
  {
    if(_sumWeight>0.0) {
      shiftValue(val, _currentAverage, _period, _halfPeriod);
    }
    _sum+=val*weight;
    _sumWeight+=weight;
    _currentAverage=_sum/_sumWeight;
    // Shift current average
    if(_currentAverage<-_halfPeriod) {
      _sum+=_sumWeight*_period;
      _currentAverage+=_period;
    } else if(_currentAverage>_halfPeriod) {
      _sum-=_sumWeight*_period;
      _currentAverage-=_period;
    }
  }

  double PeriodicStatistics::mean() const
  {
    if(_sumWeight<=0.0) {
      return std::numeric_limits<double>::quiet_NaN();
    } else {
      return _sum/_sumWeight;
    }
  }

  /*!
    Add samples to compute the variance.
    The average must be set by either calling meanAdd() for all samples
    or setAverage().
  */
  void PeriodicStatistics::varianceAdd(double val, double weight)
  {
    shiftValue(val, _currentAverage, _period, _halfPeriod);
    double c=val*weight;
    _sum+=c;
    _sum2+=c*val;
    _sumWeight+=weight;
    _sumWeight2+=weight*weight;
  }

  /*!
    Unbiased weighted sample variance.

    \sa Statistics::variance(double)
  */
  double PeriodicStatistics::variance() const
  {
    if(_sumWeight<=0.0) {
      return std::numeric_limits<double>::quiet_NaN();
    } else {
      double unbiasFactor=_sumWeight-_sumWeight2/_sumWeight;
      if(unbiasFactor>0.0) {
        return (_sum2+(_currentAverage*_sumWeight-2.0*_sum)*_currentAverage)/unbiasFactor;
      } else {
        return 0.0;
      }
    }
  }

  /*!
  */
  double PeriodicStatistics::stddev() const
  {
    double s=variance();
    if(s<0.0) s=0.0;
    return sqrt(s);
  }

  /*!
    \fn void PeriodicStatistics::shiftFunction(double& x, double mean, const double& period)
    For representing periodic function centered around \a mean.
  */

  /*!
    \fn void PeriodicStatistics::shiftValue(double& v, const double& vRef,
                                            const double& period, const double& halfPeriod)

    For shifting values
  */

  void PeriodicStatistics::test()
  {
    Complex csum, cval;
    Random rand;
    Statistics ref;
    Statistics normalStat;
    PeriodicStatistics periodicStat(180);
    for(int i=0; i<100; i++) {
      double val=rand.normal(-85, 20);
      double perval=val;
      shiftValue(perval, 180, 90);
      ref.add(val);
      normalStat.add(perval);
      cval.setUnitExp(perval/180*(2*M_PI));
      csum+=cval;
      periodicStat.meanAdd(perval);
      double corrRef=ref.mean();
      shiftValue(corrRef, 180, 90);
      printf("%i %lf %lf %lf %lf %lf %lf\n",
             i+1,
             val,
             corrRef,
             ref.stddev(),
             normalStat.mean(),
             csum.phase()*180/(2*M_PI),
             periodicStat.mean());
    }
  }

} // namespace QGpCoreMath

