/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-10-01
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef NEWTONOPTIMIZATION1D_H
#define NEWTONOPTIMIZATION1D_H

#include <QGpCoreTools.h>

#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT NewtonOptimization1D
  {
  public:
    NewtonOptimization1D();
    ~NewtonOptimization1D();

    void setRelativePrecision(double p) {_relativePrecision=p;}
    double relativePrecision() const {return _relativePrecision;}

    void setAbsolutePrecision(double p) {_absolutePrecision=p;}
    double absolutePrecision() const {return _absolutePrecision;}

    template <class Function, class Attributes>
    bool minimize(Attributes& s0, const Function * f, double defaultStep) const;
    template <class Function, class Attributes>
    bool maximize(Attributes& s0, const Function * f, double defaultStep) const;

    template <class Function, class Attributes>
    void curves(const Function * f, double minX, double maxX, double curveStepX) const;
    template <class Function, class Attributes>
    void step(const Function * f, double minX, double maxX, double curveStepX, double optimizationdefaultStep) const;
    template <class Function, class Attributes>
    void test(const Function * f, double minX, double maxX, double curveStepX, double optimizationdefaultStep) const;
  private:
    inline bool atEnd(double x, double dx) const;
    template <class Attributes> inline double step(const Attributes& a, double oldDx, double defaultStep) const;
    double _relativePrecision, _absolutePrecision;
  };

  inline bool NewtonOptimization1D::atEnd(double x, double dx) const
  {
    if(_relativePrecision>0.0) {
      return fabs(dx)<fabs(x*_relativePrecision);
    } else {
      return fabs(dx)<_absolutePrecision;
    }
  }

  template <class Attributes>
  inline double NewtonOptimization1D::step(const Attributes& a, double oldDx, double defaultStep) const
  {
    double dx=-a.slope/a.concavity;
    if(dx>defaultStep) {      // Limitation of the step size to avoid jumping to other peaks
      if(oldDx>defaultStep) {
        dx=defaultStep;
      } else {
        dx=fabs(oldDx);
      }
    } else if(dx<-defaultStep) {
      if(oldDx<-defaultStep) {
        dx=-defaultStep;
      } else {
        dx=-fabs(oldDx);
      }
    }
    return dx;
  }

  template <class Function, class Attributes>
  bool NewtonOptimization1D::minimize(Attributes& a0, const Function * f, double defaultStep) const
  {
    Attributes aSwap;
    Attributes * a1, * a2;
    a1=&a0;
    a2=&aSwap;
    int evaluationCount=2;
    f->setFunctionValue(*a1);
    f->setFunctionDerivatives(*a1);
    double dx;
    if(a1->slope>=0.0) {        // dx must be initialized for step()
      dx=-defaultStep;
    } else {
      dx=defaultStep;
    }
    while(true) {
      if(a1->concavity<=0.0) {  // We are not in the attraction zone of a minimum
        a2->x=a1->x+dx;
        evaluationCount++;
        f->setFunctionValue(*a2);
        while(a1->value>a2->value) { // Go ahead while it is decreasing
          qSwap(a1, a2);
          a2->x=a1->x+dx;
          evaluationCount++;
          f->setFunctionValue(*a2);
        }
        f->setFunctionValue(*a1);   // Evaluation at a2 might have destroyed intermediate results required for derivatives
        f->setFunctionDerivatives(*a1);
        if(a1->concavity<=0.0) {    // Still not the correct concavity
          if(a1->slope>=0.0) {      // Necessarily a minimum around a1.x but in which direction?
            dx=-defaultStep;
          } else {
            dx=defaultStep;
          }
          while(true) {             // Reduce step size to get to the correct concativity
            dx*=0.5;
            a2->x=a1->x+dx;
            evaluationCount++;
            f->setFunctionValue(*a2);
            if(a1->value>a2->value) {
              evaluationCount++;
              f->setFunctionDerivatives(*a2);
              qSwap(a1, a2);
              if(a1->concavity>0.0) {
                break;
              }
              if(a1->slope>=0.0) {  // Direction may have changed
                dx=-fabs(dx);
              } else {
                dx=fabs(dx);
              }
            }
            if(evaluationCount>100) {
              App::log("More than 100 evaluations in NewtonOptimization1D::minimize(), aborting\n");
              return false;
            }
          }
        }
      } else {
        dx=step(*a1, dx, defaultStep);
        while(true) {             // We are supposed to be in the attraction zone of a minimum
          a2->x=a1->x+dx;         // but an inflexion point might not be excluded
          evaluationCount++;      // The step must decrease the function value
          f->setFunctionValue(*a2);
          while(a1->value<a2->value) {
            dx*=0.5;
            a2->x=a1->x+dx;
            evaluationCount++;
            f->setFunctionValue(*a2);
          }
          if(atEnd(a2->x, dx)) {  // Smallest step reached, positive concavity ensured
            if(a2!=&a0) {
              a0=aSwap;
            }
            a0.evaluationCount=evaluationCount;
            return true;
          }
          evaluationCount++;     // And the concavity must remain positive
          f->setFunctionDerivatives(*a2);
          // a2 value is lower than a1 value, hence a2 is closer to the minimum than a1.
          // If the concavity in a2 is not positive anymore, an inflection point has just
          // been passed. Continue decreasing, hence go back to first step.
          if(a2->concavity<=0.0) {
            qSwap(a1, a2);
            if(a1->slope>=0.0) {        // reinitialize step
              dx=-defaultStep;
            } else {
              dx=defaultStep;
            }
            break;
          }
          dx=step(*a2, dx, defaultStep);
          qSwap(a1, a2);
          if(evaluationCount>100) {
            App::log("More than 100 evaluations in NewtonOptimization1D::minimize(), aborting\n");
            return false;
          }
        }
      }
    }
  }

  template <class Function, class Attributes>
  bool NewtonOptimization1D::maximize(Attributes& a0, const Function * f, double defaultStep) const
  {
    Attributes aSwap;
    Attributes * a1, * a2;
    a1=&a0;
    a2=&aSwap;
    int evaluationCount=2;
    f->setFunctionValue(*a1);
    f->setFunctionDerivatives(*a1);
    double dx;
    if(a1->slope>=0.0) {        // dx must be initialized for step()
      dx=defaultStep;
    } else {
      dx=-defaultStep;
    }
    while(true) {
      if(a1->concavity>=0.0) {  // We are not in the attraction zone of a maximum
        a2->x=a1->x+dx;
        evaluationCount++;
        f->setFunctionValue(*a2);
        while(a1->value<a2->value) { // Go ahead while it is increasing
          qSwap(a1, a2);
          a2->x=a1->x+dx;
          evaluationCount++;
          f->setFunctionValue(*a2);
        }
        f->setFunctionValue(*a1);   // Evaluation at a2 might have destroyed intermediate results required for derivatives
        f->setFunctionDerivatives(*a1);
        if(a1->concavity>=0.0) {    // Still not the correct concavity
          if(a1->slope>=0.0) {      // Necessarily a maximum around a1.x but in which direction?
            dx=defaultStep;
          } else {
            dx=-defaultStep;
          }
          while(true) {             // Reduce step size to get to the correct concativity
            dx*=0.5;
            a2->x=a1->x+dx;
            evaluationCount++;
            f->setFunctionValue(*a2);
            if(a1->value<a2->value) {
              evaluationCount++;
              f->setFunctionDerivatives(*a2);
              qSwap(a1, a2);
              if(a1->concavity<0.0) {
                break;
              }
              if(a1->slope>=0.0) {  // Direction may have changed
                dx=fabs(dx);
              } else {
                dx=-fabs(dx);
              }
            }
            if(evaluationCount>100) {
              App::log("More than 100 evaluations in NewtonOptimization1D::maximize(), aborting\n");
              return false;
            }
          }
        }
      } else {
        dx=step(*a1, dx, defaultStep);
        while(true) {             // We are supposed to be in the attraction zone of a maximum
          a2->x=a1->x+dx;         // but an inflexion point might not be excluded
          evaluationCount++;      // The step must increase the function value
          f->setFunctionValue(*a2);
          while(a1->value>a2->value) { // Step is too large, went over the maximum
            dx*=0.5;
            a2->x=a1->x+dx;
            evaluationCount++;
            f->setFunctionValue(*a2);
          }
          if(atEnd(a2->x, dx)) {  // Smallest step reached, negative concavity ensured
            if(a2!=&a0) {
              a0=aSwap;
            }
            a0.evaluationCount=evaluationCount;
            return true;
          }
          evaluationCount++;
          f->setFunctionDerivatives(*a2);
          // a2 value is higher than a1 value, hence a2 is closer to the maximum than a1.
          // If the concavity in a2 is not negative anymore, an inflection point has just
          // been passed. Continue increasing, hence go back to first step.
          if(a2->concavity>=0.0) {
            qSwap(a1, a2);
            if(a1->slope>=0.0) {        // reinitialize step
              dx=defaultStep;
            } else {
              dx=-defaultStep;
            }
            break;
          }
          dx=step(*a2, dx, defaultStep);
          qSwap(a1, a2);
          if(evaluationCount>100) {
            App::log("More than 100 evaluations in NewtonOptimization1D::maximize(), aborting\n");
            return false;
          }
        }
      }
    }
  }

  template <class Function, class Attributes>
  void NewtonOptimization1D::curves(const Function * f, double minX, double maxX, double curveStepX) const
  {
    Attributes a;

    QFile fout("/tmp/newton-curve");
    fout.open(QIODevice::WriteOnly);
    QTextStream sout(&fout);
    sout << "# value" << Qt::endl;
    for(a.x=minX; a.x<=maxX; a.x+=curveStepX) {
      f->setFunctionValue(a);
      sout << a.x << " " << a.value << Qt::endl;
    }
    sout << "# slope" << Qt::endl;
    for(a.x=minX; a.x<=maxX; a.x+=curveStepX) {
      f->setFunctionValue(a);
      f->setFunctionDerivatives(a);
      sout << a.x << " " << a.slope << Qt::endl;
    }
    sout << "# concavity" << Qt::endl;
    for(a.x=minX; a.x<=maxX; a.x+=curveStepX) {
      f->setFunctionValue(a);
      f->setFunctionDerivatives(a);
      sout << a.x << " " << a.concavity << Qt::endl;
    }
    fout.close();
  }

  template <class Function, class Attributes>
  void NewtonOptimization1D::step(const Function * f, double minX, double maxX, double curveStepX, double optimizationdefaultStep) const
  {
    Attributes a;

    QFile fout("/tmp/newton-step");
    fout.open(QIODevice::WriteOnly);
    QTextStream sout(&fout);
    sout << "# newton step" << Qt::endl;
    for(a.x=minX; a.x<=maxX; a.x+=curveStepX) {
      f->setFunctionValue(a);
      f->setFunctionDerivatives(a);
      sout << a.x << " " << step(a, 0.0, optimizationdefaultStep) << Qt::endl;
    }
    fout.close();
  }

  template <class Function, class Attributes>
  void NewtonOptimization1D::test(const Function * f, double minX, double maxX, double curveStepX, double optimizationdefaultStep) const
  {
    Attributes a;

    QFile fout("/tmp/newton-test");
    fout.open(QIODevice::WriteOnly);
    QTextStream sout(&fout);
    sout << "# convergence" << Qt::endl;
    for(double x=0.5*minX; x<=0.5*maxX; x+=curveStepX) {
      a.x=x;
      maximize(a, f, optimizationdefaultStep);
      sout << x << " " << a.x << Qt::endl;
    }
    sout << "# number of evaluations" << Qt::endl;
    for(double x=0.5*minX; x<=0.5*maxX; x+=curveStepX) {
      a.x=x;
      maximize(a, f,optimizationdefaultStep);
      sout << x << " " << a.evaluationCount << Qt::endl;
    }
    fout.close();
  }

} // namespace QGpCoreMath

#endif // NEWTONOPTIMIZATION1D_H

