/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "FunctionSearch.h"
#include "AbstractFunction.h"

namespace QGpCoreMath {

  /*!
    \class FunctionSearch FunctionSearch.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  FunctionSearch::FunctionSearch(int dimensionCount)
    : _currentPosition(dimensionCount, new double[dimensionCount]),
      _testPosition(dimensionCount, 0.0),
      _relative(dimensionCount, true),
      _precision(dimensionCount, 0.001)
  {
    TRACE;
    ASSERT(dimensionCount>0);
    _currentPosition.setValue(0.0);
    _function=nullptr;
  }

  /*!
    Delete current function. To prevent automatic deletion, call takeFunction() before.
  */
  FunctionSearch::~FunctionSearch()
  {
    TRACE;
    delete _function;
    delete [] _currentPosition.values();
    qDeleteAll(_maxima);
  }

  /*!
    Set current function. Ownership is transfered to this object. To get it back you must
    call takeFunction().
  */
  void FunctionSearch::setFunction(AbstractFunction * f)
  {
    TRACE;
    delete _function;
    _function=f;
  }

  /*!
    Take ownership of the function object
  */
  AbstractFunction * FunctionSearch::takeFunction()
  {
    TRACE;
    AbstractFunction * f=_function;
    _function=nullptr;
    return f;
  }

  void FunctionSearch::globalMax(double absThres)
  {
    TRACE;
    ASSERT(_maxima.count()==1);
    FunctionSearchMaximum * m=_maxima.first();
    double * curPos=_currentPosition.values(); // Save main vector for current position
    _currentPosition.setValues(m->_position);  // Temporarily set as the position of the maximum
    double val=refineMax(m->_value);
    if(val<absThres) {
      delete _maxima.takeLast();
    } else {
      m->_value=val;
    }
    _currentPosition.setValues(curPos);
  }

  void FunctionSearch::localMax(int nMax, double absThres, double relThres)
  {
    TRACE;
    // Refine maxima
    double * curPos=_currentPosition.values(); // Save main vector for current position
    for(int i=_maxima.count()-1; i>=0; i--) {
      FunctionSearchMaximum * m=_maxima.at(i);
      _currentPosition.setValues(m->_position);  // Temporarily set as the position of the maximum
      double val=refineMax(m->_value);
      m->_value=val;
    }
    _currentPosition.setValues(curPos);
     // No selection: no sorting required (help also debugging)
    if(nMax==INT_MAX && absThres==0.0 && relThres==0.0) {
      return;
    }
    // Sort and filter
    std::sort(_maxima.begin(), _maxima.end(), FunctionSearchMaximum::greaterThanValue);
    // filter the ensemble of maxima
    // With an absolute value for the maximum
    if(absThres>0.0) {
      int i=_maxima.count()-1;
      while(i>=0 && _maxima.at(i)->value()<absThres) {
        delete _maxima.takeLast();
        i--;
      }
    }
    // With a value for the maximum defined as a ratio of the global max
    if(relThres>0.0 && _maxima.count()>0) {
      absThres=_maxima.at(0)->value()*relThres;
      int i=_maxima.count()-1;
      while(i>=0 && _maxima.at(i)->value()<absThres) {
        delete _maxima.takeLast();
        i--;
      }
    }
    // Remove all maxima exceeding nMax
    while(_maxima.count()>nMax) {
      delete _maxima.takeLast();
    }
  }

  /*!
    Remove all solutions that are too close.

    Might be used for a dericative based search
  */
  void FunctionSearch::unique()
  {
    TRACE;
    std::sort(_maxima.begin(), _maxima.end(), FunctionSearchMaximum::lessThanPos);
    QGpCoreTools::unique(_maxima, FunctionSearchMaximum::equalPos);
  }

  void FunctionSearch::clearMaxima()
  {
    TRACE;
    qDeleteAll(_maxima);
    _maxima.clear();
  }

  /*!
    Maxima are eventually refined by refineMax()
  */
  void FunctionSearch::addMaximum(double val)
  {
    TRACE;
    _maxima.append(new FunctionSearchMaximum(_currentPosition, val));
  }

  /*!
    By default does nothing, the current position is left unchanged.
  */
  double FunctionSearch::refineMax(double val)
  {
    return val;
  }

  void FunctionSearch::debugScan(const Vector<double>& min, const Vector<double>& max, int n)
  {
    PrivateVector<double> d(max);
    d-=min;
    d/=n;
    for(int ix=0; ix<=n; ix++) {
      _currentPosition[0]=min[0]+d[0]*ix;
      if(d.count()==1) {
        printf("%10.5lf %lg\n", _currentPosition[0], value(_currentPosition));
      } else {
        for(int iy=0; iy<=n; iy++) {
          _currentPosition[1]=min[1]+d[1]*iy;
          if(d.count()==2) {
            printf("%10.5lf %10.5lf %lg\n",
                   _currentPosition[0],
                   _currentPosition[1],
                   value(_currentPosition));
          } else {
            for(int iz=0; iz<=n; iz++) {
              _currentPosition[2]=min[2]+d[2]*iz;
              printf("%10.5lf %10.5lf %10.5lf %lg\n",
                     _currentPosition[0],
                     _currentPosition[1],
                     _currentPosition[2],
                     value(_currentPosition));
            }
          }
        }
      }
    }
  }

} // namespace QGpCoreMath
