/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-08-23
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef CURVEPROXY_H
#define CURVEPROXY_H

#include <QGpCoreTools.h>

#include "Curve.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class IrregularGrid2D;

  class QGPCOREMATH_EXPORT CurveProxy
  {
    TRANSLATIONS("CurveProxy")
  public:
    CurveProxy() {}
    virtual ~CurveProxy() {}

    virtual QString tag() const=0;
    virtual CurveProxy * clone() const=0;

    virtual int sampleCount() const=0;
    virtual void sort()=0;
    virtual void resize(int n)=0;

    virtual double minimumX() const=0;
    virtual double maximumX() const=0;
    virtual double minimumY(const CurvePointOptions * pointOptions) const=0;
    virtual double maximumY(const CurvePointOptions * pointOptions) const=0;

    virtual double x(int sample) const=0;
    virtual void setX(int sample, double value)=0;

    virtual double y(int sample, const CurvePointOptions * options) const=0;
    virtual void setY(int sample, double value, const CurvePointOptions * options)=0;

    virtual double stddev(int /*sample*/) const {return 0.0;}
    virtual void setStddev(int /*sample*/, double /*value*/) {}

    virtual double weight(int /*sample*/) const {return 1.0;}
    virtual void setWeight(int /*sample*/, double /*value*/) {}

    virtual bool isValid(int sample) const=0;
    virtual void setValid(int sample, bool v)=0;

    virtual void setName(const QString & n)=0;
    virtual QString name() const=0;

    virtual void addLog(QString s)=0;
    virtual QString log() const=0;

    virtual void cut(double min, double max, SamplingOptions options)=0;
    virtual void validate(double min, double max, bool value, SamplingOptions options)=0;
    virtual void resample(int n, double min, double max, double valX, double valY,
                          SamplingOptions options,
                          const CurvePointOptions * pointOptions)=0;
    virtual void resample(int n, double min, double max, SamplingOptions options)=0;
    virtual void average(const CurveProxy * o)=0;
    virtual void smooth(const SamplingParameters& sampling,
                        const SmoothingParameters& param)=0;
    virtual bool isInsideRange(double x) const=0;
    virtual void interpole(double x, double& y, double& dy,
                           const CurvePointOptions * options,
                           SamplingOptions xSampling,
                           SamplingOptions ySampling) const=0;
    virtual void followMaximumX(IrregularGrid2D * grid, double min, double max)=0;

    virtual void xInverse()=0;
    virtual void yInverse()=0;
    virtual void xLog()=0;
    virtual void yLog()=0;

    virtual void beginSplit(double maxX, SamplingOptions options,
                            double maxErr, int minCount);
    virtual int splitCount() const;
    virtual void getSplit(const CurveProxy * o, int index);
    virtual void endSplit();

    virtual QVariant columnValue(int sample, int col) const=0;
    virtual void setColumnValue(int sample, int col, const QVariant & value)=0;

    virtual int columnCount() const=0;
    virtual QString columnName(int col) const=0;
    virtual QString columnUnit(int col) const=0;
    virtual QString columnTitle(int col) const;

    virtual int xAxisColumn() const=0;
    virtual int xAxisInverseColumn() const=0;
    virtual int yAxisColumn() const=0;
    virtual int yAxisInverseColumn() const=0;

    virtual QStringList columnFileTypes() const=0;
    virtual VectorList<int> defaultColumnFileTypes() const=0;
    virtual VectorList<int> savedColumns() const=0;

    bool isAbove(const CurveProxy * o, const CurvePointOptions * pointOptions,
                 SamplingOptions xSampling, SamplingOptions ySampling) const;

    void setComments(QString c);
    void save(QTextStream& s);
    virtual bool parse(ColumnTextIterator& it)=0;

    QString xName() const {return columnName(xAxisColumn());}
    QString xUnit() const {return columnUnit(xAxisColumn());}
    QString xTitle() const {return columnTitle(xAxisColumn());}
    QString xInverseName() const {return columnName(xAxisInverseColumn());}
    QString xInverseUnit() const {return columnUnit(xAxisInverseColumn());}
    QString xInverseTitle() const {return columnTitle(xAxisInverseColumn());}
    QString yName() const {return columnName(yAxisColumn());}
    QString yUnit() const {return columnUnit(yAxisColumn());}
    QString yTitle() const {return columnTitle(yAxisColumn());}
    QString yInverseName() const {return columnName(yAxisInverseColumn());}
    QString yInverseUnit() const {return columnUnit(yAxisInverseColumn());}
    QString yInverseTitle() const {return columnTitle(yAxisInverseColumn());}
  };

#define CURVEPROXY_BASIC_DECLARATIONS(curveClass) \
  int sampleCount() const {return curve().count();} \
  void sort() {curve().sort();} \
  void resize(int n) {curve().resize(n);} \
  double minimumX() const; \
  double maximumX() const; \
  double minimumY(const CurvePointOptions * pointOptions) const; \
  double maximumY(const CurvePointOptions * pointOptions) const; \
  double x(int sample) const {return curve().constAt(sample).x();} \
  void setX(int sample, double value) {curve().at(sample).setX(value);} \
  bool isValid(int sample) const {return curve().isValid(sample);} \
  void setValid(int sample, bool v) {curve().setValid(sample, v);} \
  void cut(double min, double max, SamplingOptions options) { \
    curve().cut(min, max, options); \
  } \
  void validate(double min, double max, bool value, SamplingOptions options) { \
    curve().validate(min, max, value, options); \
  } \
  void resample(int n, double min, double max, double valX, double valY, \
                        SamplingOptions options, \
                        const CurvePointOptions * pointOptions) { \
    curve().resample(n, min, max, valX, valY, options, pointOptions); \
  } \
  void resample(int n, double min, double max, \
                SamplingOptions options) { \
    curve().resample(n, min, max, options); \
  } \
  void smooth(const SamplingParameters& sampling, \
              const SmoothingParameters& param) { \
    curve().smooth(sampling, param); \
  } \
  double xAt(int index) const { \
    return curve().x(index); \
  } \
  bool isInsideRange(double x) const { \
    return curve().isInsideRange(x); \
  } \
  void xInverse() { \
    curve().xInverse(); \
  } \
  void yInverse() { \
    curve().yInverse(); \
  } \
  void xLog() { \
    curve().xLog(); \
  } \
  void yLog() { \
    curve().yLog(); \
  } \
  void setCurve(curveClass * c) {_curve=c;} \
  const curveClass& curve() const {return *_curve;} \
  curveClass& curve() {return *_curve;} \


#define CURVEPROXY_ADVANCED_DECLARATIONS(className) \
  double y(int sample, const CurvePointOptions *) const; \
  void setY(int sample, double value, const CurvePointOptions *); \
  double stddev(int sample) const; \
  void setStddev(int sample, double value); \
  double weight(int sample) const; \
  void setWeight(int sample, double value); \
  void interpole(double x, double& y, double& dy, \
                 const CurvePointOptions *, \
                 SamplingOptions xSampling, \
                 SamplingOptions ySampling) const; \
  void followMaximumX(IrregularGrid2D * grid, double min, double max); \
  void average(const CurveProxy * o) { \
    curve().average(static_cast<const className *>(o)->curve()); \
  } \
  void beginSplit(double maxX, SamplingOptions options, \
                          double maxErr, int minCount); \
  int splitCount() const; \
  void getSplit(const CurveProxy * o, int index); \
  void endSplit(); \


#define CURVEPROXY_NAME_DECLARATIONS \
  virtual void setName(const QString & n) {curve().setName(n);} \
  virtual QString name() const {return curve().name();} \


#define CURVEPROXY_LOG_DECLARATIONS \
  virtual void addLog(QString s) {curve().addLog(s);} \
  virtual QString log() const {return curve().log();} \

#define CURVEPROXY_PRIVATE_DECLARATIONS(curveClass, pointClass) \
  curveClass * _curve; \
  QList<Curve<pointClass>> * _split; \

#define CURVEPROXY_BASIC_IMPLEMENTATIONS(className) \
  double className::minimumX() const { \
    int i=curve().minimumX(); \
    if(i>=0) { \
      return curve().x(i); \
    } else { \
      return std::numeric_limits<double>::infinity(); \
    } \
  } \
  double className::maximumX() const { \
    int i=curve().maximumX(); \
    if(i>=0) { \
      return curve().x(i); \
    } else { \
      return -std::numeric_limits<double>::infinity(); \
    } \
  } \
  double className::minimumY(const CurvePointOptions * pointOptions) const { \
    int i=curve().minimumY(0, pointOptions); \
    if(i>=0) { \
      return curve().y(i, pointOptions); \
    } else { \
      return std::numeric_limits<double>::infinity(); \
    } \
  } \
  double className::maximumY(const CurvePointOptions * pointOptions) const { \
    int i=curve().maximumY(0, pointOptions); \
    if(i>=0) { \
      return curve().y(i, pointOptions); \
    } else { \
      return -std::numeric_limits<double>::infinity(); \
    } \
  }

#define CURVEPROXY_ADVANCED_IMPLEMENTATIONS(className, curveClass, pointClass) \
  double className::y(int sample, const CurvePointOptions *) const { \
    const pointClass& p=curve().constAt(sample); \
    return p.mean(); \
  } \
  void className::setY(int sample, double value, \
                       const CurvePointOptions *) { \
    pointClass& p=curve().constXAt(sample); \
    p.setMean(value); \
  } \
  double className::stddev(int sample) const { \
    const pointClass& p=curve().constAt(sample); \
    return p.stddev(); \
  } \
  void className::setStddev(int sample, double value) { \
    pointClass& p=curve().constXAt(sample); \
    p.setStddev(value); \
  } \
  double className::weight(int sample) const { \
    const pointClass& p=curve().constAt(sample); \
    return p.weight(); \
  } \
  void className::setWeight(int sample, double value) { \
    pointClass& p=curve().constXAt(sample); \
    p.setWeight(value); \
  } \
  void className::interpole(double x, double& y, double& dy, \
                            const CurvePointOptions *, \
                            SamplingOptions xSampling, \
                            SamplingOptions ySampling) const { \
    const pointClass& p=_curve->interpole(x, xSampling, ySampling); \
    y=p.mean(); \
    dy=p.stddev(); \
  } \
  void className::followMaximumX(IrregularGrid2D * grid, double min, double max) { \
    grid->followMaximumX<curveClass, pointClass>(*_curve, min, max); \
  } \
  void className::beginSplit(double maxX, SamplingOptions options, double maxErr, int minCount) \
  { \
    ASSERT(!_split); \
    _split=new QList<Curve<pointClass>>; \
    *_split=curve().split(maxX, options, maxErr, minCount); \
  } \
  int className::splitCount() const \
  { \
    ASSERT(_split); \
    return _split->count(); \
  } \
  void className::getSplit(const CurveProxy * o, int index) \
  { \
    QList<Curve<pointClass>> * split; \
    split=static_cast<const className *>(o)->_split; \
    ASSERT(split); \
    curve()=split->at(index); \
  } \
  void className::endSplit() \
  { \
    delete _split; \
    _split=nullptr; \
  }


} // namespace QGpCoreMath

#endif // CURVEPROXY_H

