/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-11-07
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "BSpline.h"

namespace QGpCoreMath {

  /*!
    \class BSpline BSpline.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing

    Source
      https://fr.wikipedia.org/wiki/B-spline
      https://en.wikipedia.org/wiki/B-spline
  */
  BSpline::BSpline(int controlPointCount, int order, double mint, double maxt)
  {
    ASSERT(order>=0);
    ASSERT(controlPointCount>0);
    _controlPointCount=controlPointCount;
    _controlPoints=new Point2D[_controlPointCount];
    _n=order;
    int m=_n+_controlPointCount;
    _knots=new double[m+1];
    int i;
    for(i=0; i<=_n; i++) {
      _knots[i]=mint;
    }
    double step=(maxt-mint)/(_controlPointCount-_n);
    for(; i<=_controlPointCount; i++) {
      _knots[i]=(i-_n)*step+mint;
    }
    for(; i<=m; i++) {
      _knots[i]=maxt;
    }
  }

  /*!
    Description of destructor still missing
  */
  BSpline::~BSpline()
  {
    delete [] _controlPoints;
    delete [] _knots;
  }

  double BSpline::minimumT() const
  {
    return _knots[_n];
  }

  double BSpline::maximumT() const
  {
    return _knots[_controlPointCount];
  }

  /*!
    Source
      https://fr.wikipedia.org/wiki/B-spline
      https://en.wikipedia.org/wiki/B-spline
  */
  Point2D BSpline::value(double t) const
  {
    Point2D s, si;
    // m+1 knots, m-n control points
    for(int i=controlPointCount()-1; i>=0; i--) {
      si=controlPoint(i);
      si*=base(i, _n, t);
      s+=si;
    }
    return s;
  }

  inline double BSpline::weight(int j, int n, double t) const
  {
    if(_knots[j+n]!=_knots[j]) {
      return (t-_knots[j])/(_knots[j+n]-_knots[j]);
    } else {
      return 0.0;
    }
  }

  /*!
    Source
      https://fr.wikipedia.org/wiki/B-spline
      https://en.wikipedia.org/wiki/B-spline
  */
  double BSpline::base(int j, int n, double t) const
  {
    if(n==0) {
      if(t>=_knots[j] && t<_knots[j+1]) {
        return 1.0;
      } else {
        return 0.0;
      }
    } else {
      return weight(j, n, t)*base(j, n-1, t)+(1.0-weight(j+1, n, t))*base(j+1, n-1, t);
    }
  }

} // namespace QGpCoreMath

