/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-10-07
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef Angle_H
#define Angle_H

#include <math.h>
#include <QtCore>

#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  class Point2D;

  class QGPCOREMATH_EXPORT Angle
  {
  public:
    inline Angle();
    Angle(double dx, double dy) {set(dx, dy);}
    inline Angle(double dx, double dy, double r) {set(dx,dy,r);}

    enum Mode {Degrees, Radians, Gradians};

    bool operator<(const Angle& o) const {return _rad<o._rad;}
    bool operator==(const Angle& o) const {return _rad==o._rad;}
    void operator+=(const Angle& o);
    bool isNull() const {return _rad==0.0;}

    inline void set(double dx, double dy);
    inline void set(double dx, double dy, double r);
    void setGeographicDegrees(double degrees) {setDegrees(90.0-degrees);}
    void setGeographicRadians(double radians) {setRadians(0.5*M_PI-radians);}
    void setGeographicGradians(double gradians) {setGradians(100.0-gradians);}
    inline void setDegrees(double degrees);
    inline void setRadians(double radians);
    inline void setGradians(double gradians);
    void setValue(Mode m, double v);
    void setGeographicValue(Mode m, double v);

    inline void initDegrees();
    inline void initRadians();

    double degrees() const {return _deg;}
    double radians() const {return _rad;}
    double gradians() const {return degreesToGradians(_deg);}
    double value(Mode m) const;
    double cos() const {return _cos;}
    double sin() const {return _sin;}
    double tan() const {return _sin/_cos;}
    double cotan() const {return _cos/_sin;}
    inline void chSign();
    inline void mirror();
    void normalize();

    static double degreesToDMS(double angle);
    static double DMSToDegrees(double angle);
    static double degreesToDM(double angle);
    static double DMToDegrees(double angle);
    static inline double radiansToDegrees(double angle);
    static inline double degreesToRadians(double angle);
    static inline double gradiansToDegrees(double angle);
    static inline double degreesToGradians(double angle);
    static inline double radiansToGradians(double angle);
    static inline double gradiansToRadians(double angle);

    static inline double geographicToMath(double radians);
    static inline double mathToGeographic(double radians);
    static double aroundZero(double radians);

    static double canonicalDegrees(double angle);
    static double canonicalRadians(double angle);
    static double canonicalGradians(double angle);
    static double differenceDegrees(double a1, double a2);
    static double differenceRadians(double a1, double a2);
    static double differenceGradians(double a1, double a2);

    static inline bool isInsideDegrees(double angle, double min, double max);
    static inline bool isInsideRadians(double angle, double min, double max);
    static inline bool isInsideGradians(double angle, double min, double max);

    static double fromString(QString angle);
  private:
    double _deg;
    double _rad;
    double _cos;
    double _sin;
  };

  inline double Angle::radiansToDegrees(double angle)
  {
    return angle*(180.0/M_PI);
  }

  inline double Angle::degreesToRadians(double angle)
  {
    return angle*(M_PI/180.0);
  }

  inline double Angle::gradiansToDegrees(double angle)
  {
    return angle*(180.0/200.0);
  }

  inline double Angle::degreesToGradians(double angle)
  {
    return angle*(200.0/180.0);
  }

  inline double Angle::radiansToGradians(double angle)
  {
    return angle*(200.0/M_PI);
  }

  inline double Angle::gradiansToRadians(double angle)
  {
    return angle*(M_PI/200.0);
  }

  inline double Angle::geographicToMath(double radians)
  {
    return radians<=0.5*M_PI ? 0.5*M_PI-radians : 2.5*M_PI-radians;
  }

  inline double Angle::mathToGeographic(double radians)
  {
    return radians<=0.5*M_PI ? 0.5*M_PI-radians : 2.5*M_PI-radians;
  }

  inline Angle::Angle()
  {
    _deg=0.0;
    _rad=0.0;
    _cos=1.0;
    _sin=0.0;
  }

  inline void Angle::set(double dx, double dy)
  {
    _deg=0.0;
    _rad=0.0;
    double r=::sqrt(dx*dx+dy*dy);
    if(r>0.0) {
      _cos=dx/r;
      _sin=dy/r;
    } else {
      _cos=1.0;
      _sin=0.0;
    }
  }

  inline void Angle::set(double dx, double dy, double r)
  {
    _deg=0.0;
    _rad=0.0;
    if(r>0.0) {
      _cos=dx/r;
      _sin=dy/r;
    } else {
      _cos=1.0;
      _sin=0.0;
    }
  }

  inline void Angle::initDegrees()
  {
    _deg=radiansToDegrees(_rad);
  }

  inline void Angle::initRadians()
  {
    _rad=::atan2(_sin, _cos);
  }

  inline void Angle::setDegrees(double degrees)
  {
    _deg=degrees;
    _rad=degreesToRadians(_deg);
    _cos=::cos(_rad);
    _sin=::sin(_rad);
  }

  inline void Angle::setRadians(double radians)
  {
    _rad=radians;
    _cos=::cos(_rad);
    _sin=::sin(_rad);
  }

  inline void Angle::setGradians(double gradians)
  {
    _deg=gradiansToDegrees(gradians);
    _rad=degreesToRadians(_deg);
    _cos=::cos(_rad);
    _sin=::sin(_rad);
  }

  inline void Angle::chSign()
  {
    _deg=-_deg;
    _rad=-_rad;
    _sin=-_sin;
  }

  inline void Angle::mirror()
  {
    _cos=-_cos;
    _sin=-_sin;
    if(_deg>180.0) _deg-=180; else _deg+=180.0;
    if(_rad>M_PI) _rad-=M_PI; else _rad+=M_PI;
  }

  inline bool Angle::isInsideDegrees(double angle, double min, double max)
  {
    // If testing only for negative angles, two conditions will be systematically checked
    // If testing directly with min which is between 0 and 2*pi, there a maximum of two
    // tests to achive, but only one if lucky enough.
    if(angle<min) {
      angle+=360.0;
      if(angle<min) {
        angle+=360.0;
      }
    }
    return angle<max;
  }

  inline bool Angle::isInsideRadians(double angle, double min, double max)
  {
    // If testing only for negative angles, two conditions will be systematically checked
    // If testing directly with min which is between 0 and 2*pi, there a maximum of two
    // tests to achive, but only one if lucky enough.
    if(angle<min) {
      angle+=2.0*M_PI;
      if(angle<min) {
        angle+=2.0*M_PI;
      }
    }
    return angle<max;
  }

  inline bool Angle::isInsideGradians(double angle, double min, double max)
  {
    // If testing only for negative angles, two conditions will be systematically checked
    // If testing directly with min which is between 0 and 2*pi, there a maximum of two
    // tests to achive, but only one if lucky enough.
    if(angle<min) {
      angle+=400.0;
      if(angle<min) {
        angle+=400.0;
      }
    }
    return angle<max;
  }

} // namespace QGpCoreMath

#endif // ANGLE_H
