/***************************************************************************
**
**  This file is part of QGpCompatibility.
**
**  QGpCompatibility is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCompatibility is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-02-11
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AUTOCORRDATA_H
#define AUTOCORRDATA_H

#include "QGpCompatibilityDLLExport.h"
#include "CompatAutocorrCurves.h"

namespace QGpCompatibility {

class QGPCOMPATIBILITY_EXPORT CompatAutocorrData :  public CompatAutocorrCurves
{
public:
  /// Default constructor
  CompatAutocorrData();
  /// Allocates an empty structure
  CompatAutocorrData(int modesCount, int radiusCount, int omegasCount);
  /// Copy constructor changing the number of modes  
  CompatAutocorrData(const CompatAutocorrData * curve, int modesCount);
  /// Set the CompatCompatAutocorrCurve as the measurements
  CompatAutocorrData(const CompatAutocorrCurves * curve);
  /// Destructor: get rid of all unused memory
  virtual ~CompatAutocorrData();
  
  // I/O functions
  // -------------
  /// Store in an inversion report
  void dataToReport(QDataStream& s);
  /// Load from an inversion report
  void reportToData(QDataStream& s);
  /// Version 3 of CompatCompatInversionReport, statistical weight of all point
  void reportToDataWeight(QDataStream& s, bool loadIt);
  
  /// Set the average of the ith measurements of radius r
  void setMean(int i, int r, int m, double mean) {_measurements[m][r][i]=mean;}
  /// Returns the ith measurements of radius r
  double measurement(int i, int r, int m) {return _measurements[m][r][i];}
  /// Set the stddev of the ith measurements of radius r
  void setStddev(int i, int r, int m, double stddev) {_stddev[m][r][i]=stddev;}
  /// Returns the stddev of the ith measurements of radius r
  double stddev(int i, int r, int m) {return _stddev[m][r][i];}
  /// Set the weith of the ith measurements of radius r
  void setWeight(int i, int r, int m, double w) {_weight[m][r][i]=w;}
  /// Returns the weith of the ith measurements of radius r
  double weight(int i, int r, int m) {return _weight[m][r][i];}
  /// Returns the measurements of radius r
  VectorList<double> * measurements(int iMode,int iRadius);
  /// Returns the measurements of frequency iOmega
  VectorList<double> * distanceMeasurements(int iMode,int iOmega);
  /// Returns the stddev of radius r
  VectorList<double> * stddev(int iMode,int iRadius);
  /// Returns the stddev of frequency iOmega
  VectorList<double> * distanceStddev(int iMode,int iOmega);
  /// Returns the weight of radius r
  VectorList<double> * weight(int iMode,int iRadius);
  
  /// Compares the measurments of two objects
  bool isSameData(const CompatAutocorrData * other) const;
  /// return the log comments
  QString & log() {return _log;}
  /// add new lines to existing comments
  void addLog(QString& s) {_log+=s+"\n";}
  ///Transfers values from the autocorrCurve o to measurements
  void valuesToData(const CompatAutocorrCurves * o);
  ///Transfers values from the autocorrCurve sub-object to measurements
  void valuesToData() {valuesToData(this);}
  
protected:
  /// Storage for measured autocorrelation curves
  double *** _measurements;  
  /// Storage for measured autocorrelation standard error
  double *** _stddev;
  /// Storage for measured autocorrelation weights
  double *** _weight;
  /// Any comment on how the measurement curve was generated
  QString _log;
  
  /// Allocates the storage for measured values
  void allocatesData();
  /// De-allocates the storage for measured values
  void deleteData();
};

} // namespace QGpCompatibility

#endif // COMPATAUTOCORRDATA_H
