/***************************************************************************
**
**  This file is part of GeopsyPyCoreWave.
**
**  GeopsyPyCoreWave is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyPyCoreWave is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-11-16
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifdef HAS_PYTHON
#define PY_SSIZE_T_CLEAN
#include <Python.h>
#define NPY_NO_DEPRECATED_API NPY_1_7_API_VERSION
#include <numpy/arrayobject.h>

#include "PythonArguments.h"
#include "PythonError.h"

/*!
  \class PythonArguments PythonArguments.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
PythonArguments::PythonArguments()
{
  TRACE;
  _h=NULL;
  _vp=NULL;
  _vs=NULL;
  _rho=NULL;
  _omega=NULL;
  _values=NULL;
  _nModes=1;
  _group=0;
  if(PyArray_API==NULL) { // Weird behavior, if not in cpp, warning at compilation
                          // and crash when using array functions.
    _import_array();
  }
}

/*!
  Description of destructor still missing
*/
PythonArguments::~PythonArguments()
{
  TRACE;
  Py_XDECREF(_h);
  Py_XDECREF(_vp);
  Py_XDECREF(_vs);
  Py_XDECREF(_rho);
  Py_XDECREF(_omega);
#if NPY_API_VERSION>=0x0000000c
  PyArray_DiscardWritebackIfCopy(_values);
#endif
}

PyArrayObject * PythonArguments::toVectorArray(PyObject * obj, const QString& arrayName)
{
  PyArrayObject * array=(PyArrayObject *) PyArray_FROM_OTF(obj, NPY_DOUBLE, NPY_ARRAY_IN_ARRAY);
  if(array==NULL) {
    PythonError::setString(tr("object for %1 argument must be a numpy array.").arg(arrayName));
    return NULL;
  }
  if(PyArray_NDIM(array)!=1) {
    PythonError::setString(tr("dimension of %1 array must be 1.").arg(arrayName));
    return NULL;
  }
  return array;
}

bool PythonArguments::parseRayleighDispersion(PyObject * args)
{
  PyObject * argH=NULL, * argVp=NULL, * argVs=NULL, * argRho=NULL;
  PyObject * argOmega=NULL;

  if(!PyArg_ParseTuple(args, "iiOOOOO", &_nModes, &_group,
                        &argH, &argVp, &argVs, &argRho,
                        &argOmega)) {
    return false;
  }
  _h=toVectorArray(argH, tr("thickness"));
  if(!_h) {
    return false;
  }
  _vp=toVectorArray(argVp, tr("Vp"));
  if(!_vp) {
    return false;
  }
  _vs=toVectorArray(argVs, tr("Vs"));
  if(!_vs) {
    return false;
  }
  _rho=toVectorArray(argRho, tr("density"));
  if(!_rho) {
    return false;
  }
  _omega=toVectorArray(argOmega, tr("omega"));
  if(!_omega) {
    return false;
  }
  return checkLayerNumber();
}

bool PythonArguments::parseRayleighEllipticity(PyObject * args)
{
  PyObject * argH=NULL, * argVp=NULL, * argVs=NULL, * argRho=NULL;
  PyObject * argOmega=NULL;

  if(!PyArg_ParseTuple(args, "iOOOOO", &_nModes,
                        &argH, &argVp, &argVs, &argRho,
                        &argOmega)) {
    return false;
  }
  _h=toVectorArray(argH, tr("thickness"));
  if(!_h) {
    return false;
  }
  _vp=toVectorArray(argVp, tr("Vp"));
  if(!_vp) {
    return false;
  }
  _vs=toVectorArray(argVs, tr("Vs"));
  if(!_vs) {
    return false;
  }
  _rho=toVectorArray(argRho, tr("density"));
  if(!_rho) {
    return false;
  }
  _omega=toVectorArray(argOmega, tr("omega"));
  if(!_omega) {
    return false;
  }
  return checkLayerNumber();
}

bool PythonArguments::parseLove(PyObject * args)
{
  PyObject * argH=NULL, * argVs=NULL, * argRho=NULL;
  PyObject * argOmega=NULL;

  if(!PyArg_ParseTuple(args, "iiOOOO", &_nModes, &_group,
                       &argH, &argVs, &argRho,
                       &argOmega)) {
    return false;
  }

  _h=toVectorArray(argH, tr("thickness"));
  if(!_h) {
    return false;
  }
  _vs=toVectorArray(argVs, tr("Vs"));
  if(!_vs) {
    return false;
  }
  _rho=toVectorArray(argRho, tr("density"));
  if(!_rho) {
    return false;
  }
  _omega=toVectorArray(argOmega, tr("omega"));
  if(!_omega) {
    return false;
  }
  return checkLayerNumber();
}

bool PythonArguments::checkLayerNumber() const
{
  App::log(1, tr("nSamples=%1, nModes=%2, group=%3\n").arg(nSamples()).arg(_nModes).arg(_group));
  npy_intp * dims;
  dims=PyArray_DIMS(_h);
  int nLayers=dims[0]+1;
  return checkLayerNumber(_vp, nLayers, tr("Vp")) &&
         checkLayerNumber(_vs, nLayers, tr("Vs")) &&
         checkLayerNumber(_rho, nLayers, tr("density"));
}

bool PythonArguments::checkLayerNumber(PyArrayObject * array, int nLayers, const QString& arrayName) const
{
  if(array!=NULL) {
    npy_intp * dims;
    dims=PyArray_DIMS(array);
    if(dims[0]!=nLayers) {
      PythonError::setString(tr("number of elements in %1 array must be one greater than that in the thickness array.")
                                 .arg(arrayName));
      return false;
    } else {
      return true;
    }
  } else {
    return true;
  }
}

int PythonArguments::nLayers() const
{
  npy_intp * dims;
  dims=PyArray_DIMS(_h);
  return dims[0]+1;
}

int PythonArguments::nSamples() const
{
  npy_intp * dims;
  dims=PyArray_DIMS(_omega);
  return dims[0];
}

VectorList<double> PythonArguments::omegaList() const
{
  int n=nSamples();
  VectorList<double> x(n);
  double * omega=(double *)PyArray_DATA(_omega);
  for(int i=0; i<n; i++) {
    x[i]=omega[i];
    App::log(2, tr("frequency[%1]=%2 Hz\n").arg(i).arg(x[i]/(2*M_PI)));
  }
  return x;
}

void PythonArguments::setValues(ModalStorage& storage)
{
  int nm=nModes();
  int ns=nSamples();

  npy_intp * dims;
  dims=new npy_intp[2];
  dims[0]=nm;
  dims[1]=ns;
  _values=(PyArrayObject *) PyArray_NewFromDescr(&PyArray_Type,
                                                 PyArray_DescrFromType(NPY_DOUBLE),
                                                 2, dims, NULL, NULL, 0, NULL);
  delete [] dims;

  double * values=(double *)PyArray_DATA(_values);
  for(int iMode=0; iMode<nm; iMode++) {
    App::log(2, tr("Mode=%1\n").arg(iMode));
    const RealValue * mode=storage.mode(iMode);
    for(int iSample=0; iSample<ns; iSample++) {
      const RealValue& v=mode[iSample];
      if(v.isValid()) {
        *(values++)=v.value();
        APP_LOG(2, tr("frequency[%1]=%2 Hz, value[%1]=%3\n")
                       .arg(iSample)
                       .arg(storage.x(iSample)/(2*M_PI))
                       .arg(v.value()));
      } else {
        *(values++)=std::numeric_limits<double>::quiet_NaN();
      }
    }
  }
}

#endif // HAS_PYTHON
