/*****
**  Created : 2023-11-14
*****/

#ifdef HAS_PYTHON
#define PY_SSIZE_T_CLEAN
#include <Python.h>
#define NPY_NO_DEPRECATED_API NPY_1_7_API_VERSION
#include <numpy/arrayobject.h>

#include "GeopsyPyCoreWaveVersion.h"
#include "GeopsyPyCoreWaveInstallPath.h"
#include "PythonArguments.h"
#include "PythonError.h"

PACKAGE_INFO("GeopsyPyCoreWave", GEOPSYPYCOREWAVE);

static PyObject * GeopsyPyCoreWave_setVerbosity(PyObject *, PyObject * args)
{
  int level;
  if (!PyArg_ParseTuple(args, "i", &level)) {
    return NULL;
  }
  if(level<0) {
    PluginCoreApplication::instance()->freezeStream(true);
  } else {
    AbstractStream::setApplicationVerbosity(level);
  }
  Py_RETURN_NONE;
}

static PyObject * GeopsyPyCoreWave_RayleighDispersionCurve(PyObject *, PyObject * args)
{
  PythonArguments p;
  if(!p.parseRayleighDispersion(args)) {
    return NULL;
  }

  QGpCoreWave::Seismic1DModel model(p.nLayers(), p.hData(),
                                    p.vpData(), p.vsData(), p.rhoData());
  App::log(1, model.toString());
  if(model.initCalculation()) {
    QGpCoreWave::Rayleigh rayleigh(&model);
    VectorList<double> x=p.omegaList();
    QGpCoreWave::Dispersion dispersion(p.nModes(), &x);
    dispersion.calculate(&rayleigh, nullptr);
    if(p.group()!=0) {
      dispersion.setGroupSlowness();
    }
    p.setValues(dispersion);
    return (PyObject *) p.values();
  } else {
    PythonError::setString(tr("check model parameters."));
    return NULL;
  }
}

static PyObject * GeopsyPyCoreWave_RayleighEllipticityCurve(PyObject *, PyObject * args)
{
  PythonArguments p;
  if(!p.parseRayleighEllipticity(args)) {
    return NULL;
  }

  QGpCoreWave::Seismic1DModel model(p.nLayers(), p.hData(),
                                    p.vpData(), p.vsData(), p.rhoData());
  App::log(1, model.toString());
  if(model.initCalculation()) {
    QGpCoreWave::Rayleigh rayleigh(&model);
    VectorList<double> x=p.omegaList();
    QGpCoreWave::Dispersion dispersion(p.nModes(), &x);
    dispersion.setPrecision(1e-10);
    QGpCoreWave::Ellipticity ell(p.nModes(), &x);
    dispersion.calculate(&rayleigh, &ell);
    if(p.group()!=0) {
      dispersion.setGroupSlowness();
    }
    p.setValues(ell);
    return (PyObject *) p.values();
  } else {
    PythonError::setString(tr("check model parameters."));
    return NULL;
  }
}

static PyObject * GeopsyPyCoreWave_LoveDispersionCurve(PyObject *, PyObject * args)
{
  PythonArguments p;
  if(!p.parseLove(args)) {
    return NULL;
  }

  QGpCoreWave::Seismic1DModel model(p.nLayers(), p.hData(),
                                    p.vsData(), p.rhoData());
  App::log(1, model.toString());
  if(model.initCalculation()) {
    QGpCoreWave::Love love(&model);
    VectorList<double> x=p.omegaList();
    QGpCoreWave::Dispersion dispersion(p.nModes(), &x);
    dispersion.calculate(&love, nullptr);
    if(p.group()!=0) {
      dispersion.setGroupSlowness();
    }
    p.setValues(dispersion);
    return (PyObject *) p.values();
  } else {
    PythonError::setString(tr("check model parameters."));
    return NULL;
  }
}

static PyMethodDef GeopsyPyCoreWaveMethods[] = {
    {"setVerbosity", GeopsyPyCoreWave_setVerbosity, METH_VARARGS,
     "Set verbosity level (-1=no ouput, 0=normal, 1=more verbose, 2=even more,...)."},
    {"rayleighDispersionCurve", GeopsyPyCoreWave_RayleighDispersionCurve, METH_VARARGS,
     "rayleighDispersionCurve(nModes, group, h array, Vp array, Vs array, rho array, omega array): computation of Rayleigh dispersion curves from a 1D layered model."},
    {"rayleighEllipticityCurve", GeopsyPyCoreWave_RayleighEllipticityCurve, METH_VARARGS,
     "rayleighElliptiictyCurve(nModes, h array, Vp array, Vs array, rho array, omega array): computation of Rayleigh ellipticity curves from a 1D layered model."},
    {"loveDispersionCurve", GeopsyPyCoreWave_LoveDispersionCurve, METH_VARARGS,
     "rayleighDispersionCurve(nModes, h array, Vs array, rho array, omega array): computation of Love dispersion curves from a 1D layered model."},
    {NULL, NULL, 0, NULL}
};

static struct PyModuleDef GeopsyPyCoreWaveModule = {
    PyModuleDef_HEAD_INIT,
    "GeopsyPyCoreWave",     // name of module
    NULL,               // module documentation, may be NULL
    -1,                 // size of per-interpreter state of the module,
    // or -1 if the module keeps state in global variables.
    GeopsyPyCoreWaveMethods,
    NULL,
    NULL,
    NULL,
    NULL
};

PyMODINIT_FUNC PyInit_GeopsyPyCoreWave(void)
{
  if(!CoreApplication::instance()) {
    new PluginCoreApplication;
  }

  PyObject* mod=PyModule_Create(&GeopsyPyCoreWaveModule);
  import_array();

  if(!PythonError::instance()->init(mod)) {
    return NULL;
  }

  return mod;
}

#endif // HAS_PYTHON
