/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-09-20
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include <QGpCoreTools.h>
#include "RelativePositions.h"

namespace GeopsyGui {

  RelativePositionsDelegate::RelativePositionsDelegate(QObject *parent)
      : ItemDelegate(parent)
  {
    TRACE;
  }

  QWidget *RelativePositionsDelegate::createEditor(QWidget *parent, const QStyleOptionViewItem &,
                                                   const QModelIndex & index) const
  {
    if(index.column()<2) {
      QComboBox * w=createComboBox(parent);
      w->addItems(_stationList);
      connect(w, SIGNAL(activated(int)), this, SLOT(commitWidget()));
      return w;
    } else {
      QLineEdit * w=createLineEdit(parent);
      w->setReadOnly(true);
      return w;
    }
  }

  /*
    Constructs a RelativePositions as a child of 'parent', with the
    name 'name' and widget flags set to 'f'.
   */
  RelativePositions::RelativePositions(QWidget *parent, Qt::WindowFlags f)
      : QWidget(parent, f)
  {
    TRACE;
    setupUi(this);

    QStringList hLabels;
    hLabels << tr("From") << tr("To") << tr("Distance") << tr("Azimuth");
    mainTable->setHorizontalHeaderLabels (hLabels);
    mainTable->setSelectionMode(QAbstractItemView::SingleSelection);
    mainTable->setEditTriggers(QAbstractItemView::AllEditTriggers);
    RelativePositionsDelegate * delegate=new RelativePositionsDelegate(this);
    mainTable->setItemDelegate (delegate);
    // signals and slots connections
    connect(addBut, SIGNAL(clicked()), this, SLOT(addBut_clicked()));
    connect(removeBut, SIGNAL(clicked()), this, SLOT(removeBut_clicked()));
    connect(mainTable, SIGNAL(itemChanged (QTableWidgetItem *)),
             this, SLOT(updateCurrentDistAz(QTableWidgetItem *)));
  }

  /*
   *  Destroys the object and frees any allocated resources
   */
  RelativePositions::~RelativePositions()
  {
    TRACE;
    // no need to delete child widgets, Qt does it all for us
  }

  void RelativePositions::setStationList(const QList<NamedPoint> * stations)
  {
    TRACE;
    _stations=stations;
    RelativePositionsDelegate * delegate=static_cast<RelativePositionsDelegate *>(mainTable->itemDelegate());
    int n=_stations->count();
    delegate->clearStations();
    for(int i=0; i<n; i++) {
      delegate->addStation(_stations->at(i).name());
    }
    updateAll();
  }

  void RelativePositions::addBut_clicked()
  {
    TRACE;
    mainTable->blockSignals(true);
    int row=mainTable->rowCount();
    mainTable->setRowCount(row+1);
    QTableWidgetItem * item;
    item=new QTableWidgetItem;
    mainTable->setItem(row,0, item);
    item=new QTableWidgetItem;
    mainTable->setItem(row,1, item);
    item=new QTableWidgetItem;
    mainTable->setItem(row,2, item);
    item=new QTableWidgetItem;
    mainTable->setItem(row,3, item);
    mainTable->resizeRowToContents(row);
    mainTable->blockSignals(false);
    if(_stations->count()>0) {
      RelativePositionsDelegate * delegate=static_cast<RelativePositionsDelegate *>(mainTable->itemDelegate());
      mainTable->item(row,0)->setText(delegate->stationAt(0));
      if(_stations->count()>1) {
        mainTable->item(row,1)->setText(delegate->stationAt(1));
      }
    }
  }

  void RelativePositions::removeBut_clicked()
  {
    TRACE;
    int row=mainTable->currentRow();
    mainTable->removeRow(row);
    mainTable->setCurrentItem(mainTable->item(row, 0));
  }

  void RelativePositions::updateCurrentDistAz(QTableWidgetItem * item)
  {
    TRACE;
    int row=mainTable->row(item);
    mainTable->blockSignals(true);
    RelativePositionsDelegate * delegate=static_cast<RelativePositionsDelegate *>(mainTable->itemDelegate());
    int iFrom=delegate->indexOf(mainTable->item(row,0)->text());
    int iTo=delegate->indexOf(mainTable->item(row,1)->text());
    if(iFrom>=0 && iTo>=0) {
      const Point& p1=_stations->at(iFrom);
      const Point& p2=_stations->at(iTo);
      mainTable->item(row,2)->setText(QString::number(p1.distanceTo(p2), 'f', 2));
      double angle=Angle::mathToGeographic(p1.azimuthTo(p2));
      angle=Angle::radiansToDegrees(angle);
      mainTable->item(row,3)->setText(QString::number(angle, 'f', 3));
    }
    mainTable->blockSignals(false);
    calculateAverage();
  }

  void RelativePositions::updateAll()
  {
    TRACE;
    mainTable->blockSignals(true);
    RelativePositionsDelegate * delegate=static_cast<RelativePositionsDelegate *>(mainTable->itemDelegate());
    for(int row=mainTable->rowCount()-1; row>=0; row--) {
      int iFrom=delegate->indexOf(mainTable->item(row,0)->text());
      int iTo=delegate->indexOf(mainTable->item(row,1)->text());
      if(iFrom>=0 && iTo>=0) {
        const Point& p1=_stations->at(iFrom);
        const Point& p2=_stations->at(iTo);
        mainTable->item(row,2)->setText(QString::number(p1.distanceTo(p2), 'f', 2));
        double angle=Angle::mathToGeographic(p1.azimuthTo(p2));
        angle=Angle::radiansToDegrees(angle);
        mainTable->item(row,3)->setText(QString::number(angle, 'f', 3));
      }
    }
    mainTable->blockSignals(false);
    calculateAverage();
  }

  void RelativePositions::calculateAverage()
  {
    TRACE;
    int nRow=mainTable->rowCount();
    double d=0, az=0;
    for(int iRow=0; iRow<nRow; iRow++) {
      d+=mainTable->item(iRow, 2)->text().toDouble();
      az+=mainTable->item(iRow, 3)->text().toDouble();
    }
    d/=(double)nRow;
    az/=(double)nRow;
    averageDistance->setText(QString::number(d));
    averageAzimuth->setText(QString::number(az));
  }

} // namespace GeopsyGui
