/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-12-22
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiTools.h>
#include <SciFigs.h>
#include "ArrayMap.h"

namespace GeopsyGui {

ArrayMap::ArrayMap(QWidget * parent, AbstractNameLine * referenceLine)
    : AxisWindow(parent)
{
  TRACE;
  _background=new ImageLayer(this);
  _background->setObjectName("background");

  _mapLayer=new NameLineLayer(this);
  _mapLayer->setErrorBar(LineLayer::NoBar);
  _mapLayer->setObjectName("map");
  if(!referenceLine) {
    referenceLine=new NameLine;
  }
  referenceLine->setPen(Pen::NoLine);
  referenceLine->setSymbol(Symbol(Symbol::Circle, 2.0, Pen::NoLine,
                           Brush(Qt::black, Qt::SolidPattern)));
  _mapLayer->setReferenceLine(referenceLine);
  _mapLayer->addLine(); // Must have at least one line (expected by spac tool for instance)

  xAxis()->setTitle( "X (m)" );
  xAxis()->setSizeType(Axis::Scaled);
  xAxis()->setSizeInfo(1000);
  xAxis()->setRange(0, 100);
  yAxis()->setTitle( "Y (m)" );
  yAxis()->setSizeType(Axis::Scaled);
  yAxis()->setSizeInfo(1000);
  yAxis()->setRange(0, 100);
  updateExternalGeometry();
}

ArrayMap::~ArrayMap()
{
  TRACE;
  Settings::setSize(this, "ArrayMap"+objectName());
}

void ArrayMap::setObjectName(QString name)
{
  TRACE;
  QWidget::setObjectName(name);
  Settings::getSize(this, "ArrayMap"+name);
}

void ArrayMap::setLimitsArray(bool includeBackground)
{
  TRACE;
  Rect r;
  if(includeBackground) {
    r=graphContents()->boundingRect();
  } else {
    r=_mapLayer->boundingRect();
  }
  r.enlarge(0.05, LinearScale, LinearScale);
  setMapScale(r.x1(), r.y1(), r.x2(), r.y2());
}

void ArrayMap::setLimitsCoArray()
{
  TRACE;
  Rect r=_mapLayer->boundingRect();
  double dmax=fabs(r.x1());
  if(fabs(r.x2())>dmax) dmax=fabs(r.x2());
  if(fabs(r.y1())>dmax) dmax=fabs(r.y1());
  if(fabs(r.y2())>dmax) dmax=fabs(r.y2());
  dmax*=1.05;
  setMapScale(-dmax, 0, dmax, dmax);
}

void ArrayMap::optimizeRing(double& minR, double& maxR)
{
  TRACE;
  double newMinR=maxR, newMaxR=minR;
  Point origin(0, 0, 0);
  int n=_mapLayer->count();
  for(int i=0; i<n; i++) {
    AbstractNameLine * l=line(i);
    int np=l->count();
    for(int j=0; j<np; j++) {
      double r=origin.distanceTo(l->point(j, nullptr));
      if(r>=minR && r<=maxR) {
        if(r>newMaxR) newMaxR=r;
        if(r<newMinR) newMinR=r;
      }
    }
  }
  minR=newMinR;
  maxR=newMaxR;
}

void ArrayMap::loadBackground(const QString& fileName)
{
  TRACE;
  // Not the most elegant way but it avoid new interface functions in GraphContents
  GraphContents * gc=graphContents();
  int index=gc->layerCount();
  gc->appendLayers(fileName);
  ImageLayer * l=qobject_cast<ImageLayer *>(gc->findLayer(ImageLayer::xmlImageLayerTag, QString(), index));
  if(l) {
    while(!_background->isRemovable()) {
      _background->removeReference();
    }
    gc->removeLayer(_background);
    index--;
    for(int i=0; i<index; i++) {
      gc->moveLayerDown(l);
    }
    _background=l;
    _background->addReference();
  }
}

} // namespace GeopsyGui
