/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-11-06
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SeismicEvent.h"

namespace GeopsyCore {

  const QString SeismicEvent::xmlSeismicEventTag="SeismicEvent";

  /*!
    \class SeismicEvent SeismicEvent.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SeismicEvent::SeismicEvent()
    : _time(QDate(2000, 1, 1), 0.0, 0.0)
  {
    TRACE;
    _type=Hammer;
    _duration=0.001;
    _force=Point(0.0, 0.0, -1.0);
    _lock=0;
  }

  SeismicEvent::SeismicEvent(const SeismicEvent& o)
  {
    TRACE;
    _position=o._position;
    _utmZone=o._utmZone;
    _time=o._time;
    _duration=o._duration;
    _name=o._name;
    _type=o._type;
    _force=o._force;
    _lock=0;
  }

  SeismicEvent::~SeismicEvent()
  {
    ASSERT(_lock==0);
  }

  void SeismicEvent::operator=(const SeismicEvent& o)
  {
    TRACE;
    _position=o._position;
    _utmZone=o._utmZone;
    _time=o._time;
    _duration=o._duration;
    _name=o._name;
    _type=o._type;
    _force=o._force;
  }

  bool SeismicEvent::operator==(const SeismicEvent& o) const
  {
    return _position==o._position &&
           _utmZone==o._utmZone &&
           _time==o._time &&
           _duration==o._duration &&
           _name==o._name &&
           _type==o._type &&
           _force==o._force;
  }

  bool SeismicEvent::isSamePosition(const SeismicEvent& o) const
  {
    return _position==o._position &&
           _utmZone==o._utmZone;
  }

  QString SeismicEvent::name(int posPrec, int timePrec) const
  {
    if(_name.isEmpty()) {
      static const QString format(tr("(%1) @ %2"));
      return format
          .arg(_position.toString('f', posPrec))
          .arg(_time.toString("hh:mm:ssz", timePrec));
    } else {
      return _name;
    }
  }

  QStringList SeismicEvent::userTypes()
  {
    QStringList l;
    l << tr("Hammer")
      << tr("Explosion")
      << tr("WeightDrop")
      << tr("Steps")
      << tr("Earthquake");
    std::sort(l.begin(), l.end());
    return l;
  }

  QString SeismicEvent::userType(Type t)
  {
    TRACE;
    switch(t) {
    case Hammer:
      break;
    case Explosion:
      return tr("Explosion");
    case WeightDrop:
      return tr("WeightDrop");
    case Steps:
      return tr("Steps");
    case Earthquake:
      return tr("Earthquake");
    }
    return tr("Hammer");
  }

  SeismicEvent::Type SeismicEvent::userType(QString t, bool * ok)
  {
    TRACE;
    t=t.toLower();
    if(t==tr("hammer")) {
    } else if(t==tr("explosion")) {
      return Explosion;
    } else if(t==tr("weightdrop")) {
      return WeightDrop;
    } else if(t==tr("steps")) {
      return Steps;
    } else if(t==tr("earthquake")) {
      return Earthquake;
    } else if(ok) {
      *ok=false;
    }
    return Hammer;
  }

  ENUM_AS_STRING_BEGIN(SeismicEvent, Type)
  ENUM_AS_STRING_DATA_5(Earthquake, Explosion, Hammer, Steps, WeightDrop);
  ENUM_AS_STRING_END

  void SeismicEvent::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s, "Position", _position.toString('g', 15));
    writeProperty(s, "UtmZone", _utmZone.toString());
    writeProperty(s, "Time", _time.toString(DateTime::defaultFormat, 15));
    writeProperty(s, "Duration", _duration);
    writeProperty(s, "Name", _name);
    writeProperty(s, "Type", convertType(_type));
    writeProperty(s, "Force", _force.toString('g', 15));
  }

  XMLMember SeismicEvent::xml_member(XML_MEMBER_ARGS)
  {
    Q_UNUSED(attributes)
    Q_UNUSED(context)
    if(tag.size()>0) {
      switch (tag[0].unicode()) {
      case 'F':
        if(tag=="Force") return XMLMember(3);
        break;
      case 'N':
        if(tag=="Name") return XMLMember(5);
        break;
      case 'P':
        if(tag=="Position") return XMLMember(0);
        break;
      case 'T':
        if(tag=="Time") return XMLMember(1);
        else if(tag=="Type") return XMLMember(2);
        break;
      case 'D':
        if(tag=="Duration") return XMLMember(6);
        break;
      case 'U':
        if(tag=="UtmZone") return XMLMember(4);
        break;
      default:
        break;
      }
    }
    return XMLMember(XMLMember::Unknown);
  }

  bool SeismicEvent::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    Q_UNUSED(context)
    Q_UNUSED(tag)
    Q_UNUSED(attributes)
    bool ok=true;
    switch(memberID) {
    case 0:
      return _position.fromString(content.toStringView());
    case 1:
      return _time.fromString(content.toStringBuffer(), DateTime::defaultFormat);
    case 2: {
        Type t=convertType(content.toStringView(), ok);
        if(ok) {
          _type=t;
          return true;
        } else {
          return false;
        }
      }
    case 3:
      return _force.fromString(content.toStringView());
    case 4:
      _utmZone.fromString(content.toStringBuffer());
      return true;
    case 5:
      _name=content.toStringBuffer();
      return true;
    case 6:
      _duration=content.toDouble(ok);
      return ok;
    default:
      return false;
    }
  }

  bool SeismicEvent::lessThan(const SeismicEvent * e1, const SeismicEvent * e2)
  {
    if(e1->time()<e2->time()) {
      return true;
    } else if(e1->time()==e2->time()) {
      return e1->position()<e2->position();
    } else {
      return false;
    }
  }

} // namespace GeopsyCore

