/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-05-18
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "PickParameters.h"

namespace GeopsyCore {

  /*!
    \class PickParameters PickParameters.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  PickParameters::PickParameters()
    : AbstractParameters()
  {
    TRACE;
    _amplitudeThresholdType=RelativeAmplitudeThreshold;
    _amplitudeThresholdValue=0.5;  // 50%
    _holdOffDelay=1.0;             // seconds
   }

  /*!
    Description of destructor still missing
  */
  PickParameters::PickParameters(const PickParameters& o)
    : AbstractParameters(o)
  {
    TRACE;
    _amplitudeThresholdType=o._amplitudeThresholdType;
    _amplitudeThresholdValue=o._amplitudeThresholdValue;
    _holdOffDelay=o._holdOffDelay;
  }

  void PickParameters::setHoldOffDelay(double d)
  {
    if(d>=0.0) {
      _holdOffDelay=d;
    }
  }

  bool PickParameters::setAmplitudeThresholdType(const QString& t)
  {
    if(!t.isEmpty()) {
      QString lt=t.toLower();
      switch(lt[0].unicode()) {
      case 'a':
        if(lt=="absoluteamplitudethreshold") {
          _amplitudeThresholdType=AbsoluteAmplitudeThreshold;
          return true;
        }
        break;
      case 'n':
        if(lt=="noamplitudethreshold") {
          _amplitudeThresholdType=NoAmplitudeThreshold;
          return true;
        }
        break;
      case 'r':
        if(lt=="relativeamplitudethreshold") {
          _amplitudeThresholdType=RelativeAmplitudeThreshold;
          return true;
        }
        break;
      default:
        break;
      }
    }
    App::log(tr("Unrecognized amplitude threshold type: %1\n").arg(t));
    return false;
  }

  QString PickParameters::amplitudeThresholdTypeString() const
  {
    switch(_amplitudeThresholdType) {
    case NoAmplitudeThreshold:
      break;
    case RelativeAmplitudeThreshold:
      return "RelativeAmplitudeThreshold";
    case AbsoluteAmplitudeThreshold:
      return "AbsoluteAmplitudeThreshold";
    }
    return "NoAmplitudeThreshold";
  }

  QString PickParameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
  {
    TRACE;
    QString log;
    log+="# TYPEs: NoAmplitudeThreshold, RelativeAmplitudeThreshold, AbsoluteAmplitudeThreshold\n";
    log+=prefix+"AMPLITUDE_THRESHOLD_TYPE"+suffix+"="+amplitudeThresholdTypeString()+"\n";
    log+=prefix+"AMPLITUDE_THRESHOLD_VALUE"+suffix+"="+QString::number(_amplitudeThresholdValue)+"\n";
    log+=prefix+"HOLD_OFF_DELAY"+suffix+" (s)="+QString::number(_holdOffDelay)+"\n";
    return log;
  }

  int PickParameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
  {
    return 3+AbstractParameters::keywordCount();
  }

  void PickParameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
  {
    TRACE;
    int baseIndex=AbstractParameters::keywordCount();
    keywords.add(prefix+"AMPLITUDE_THRESHOLD_TYPE"+suffix, this, baseIndex);
    keywords.add(prefix+"AMPLITUDE_THRESHOLD_VALUE"+suffix, this, baseIndex+1);
    keywords.add(prefix+"HOLD_OFF_DELAY"+suffix, this, baseIndex+2);
  }

  bool PickParameters::setValue(PARAMETERS_SETVALUE_ARGS)
  {
    TRACE;
    bool ok=true;
    switch(index-AbstractParameters::keywordCount()) {
    case 0:
      return setAmplitudeThresholdType(value);
    case 1:
      _amplitudeThresholdValue=value.toDouble(&ok);
      return ok;
    case 2:
      setHoldOffDelay(value.toDouble(&ok));
      return ok;
    default:
      break;
    }
    return AbstractParameters::setValue(index, value, unit, keywords);
  }

} // namespace GeopsyCore

