/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-08-14
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "AsciiSignalFormatRule.h"

namespace GeopsyCore {

  /*!
    \class AsciiSignalFormatRule AsciiSignalFormatRule.h
    \brief A rule entry in the definition of Ascii signal format

    Full description of class still missing
  */

  const QString AsciiSignalFormatRule::xmlRuleTag="Rule";
  static const QString indexKey("index");

  /*!
    Constructs a basic rule: for all channels, not constant, mandatory, empty pattern, pattern index set to 1, not multiplication factor.
  */
  AsciiSignalFormatRule::AsciiSignalFormatRule()
  {
    _channel=-1; // Default to all channels
    _constant=false;
    _mandatory=true;
    _operation=Assign;
    _patternIndex=1;
    _factor=1.0;
  }

  void AsciiSignalFormatRule::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
  {
    TRACE;
    Q_UNUSED(context)
    writeProperty(s, "channel", _channel);
    XMLSaveAttributes att;
    att.add(indexKey, _data.index());
    writeProperty(s, "data", att, MetaDataFactory::instance()->baseName(_data));
    writeProperty(s, "constant", (bool)_constant);
    if(_constant) {
      writeProperty(s, "value", _value.toString());
    } else {
      writeProperty(s, "mandatory", (bool)_mandatory);
      writeProperty(s, "operation", convertOperation(_operation));
      writeProperty(s, "pattern", _pattern);
      writeProperty(s, "patternIndex", (int)_patternIndex);
    }
    writeProperty(s, "factor", _factor);
  }

  XMLMember AsciiSignalFormatRule::xml_member(XML_MEMBER_ARGS)
  {
    TRACE;
    Q_UNUSED(context)
    Q_UNUSED(attributes)
    if(tag=="channel") return XMLMember(0);
    else if(tag=="data") return XMLMember(1);
    else if(tag=="pattern") return XMLMember(2);
    else if(tag=="patternIndex") return XMLMember(3);
    else if(tag=="constant") return XMLMember(4);
    else if(tag=="mandatory") return XMLMember(5);
    else if(tag=="operation") return XMLMember(8);
    else if(tag=="value") return XMLMember(6);
    else if(tag=="factor") return XMLMember(7);
    else return XMLMember(XMLMember::Unknown);
  }

  bool AsciiSignalFormatRule::xml_setProperty(XML_SETPROPERTY_ARGS)
  {
    TRACE;
    Q_UNUSED(tag)
    Q_UNUSED(context)
    bool ok=true;
    switch(memberID) {
    case 0: _channel=content.toInt(ok); return ok;
    case 1: {
        _data=MetaDataFactory::instance()->index(content.toStringBuffer());
        XMLRestoreAttributeIterator it=attributes.find(indexKey);
        if(it!=attributes.end()) {
          _data.setIndex(it.value().toStringBuffer());
        }
        return true;
      }
    case 2: _pattern=content.toStringBuffer(); return true;
    case 3: _patternIndex=content.toInt(ok); return ok;
    case 4: _constant=content.toBool(ok); return ok;
    case 5: _mandatory=content.toBool(ok); return ok;
    case 6: _value=content.toStringBuffer(); return true;
    case 7: _factor=content.toDouble(ok); return ok;
    case 8: setOperation(convertOperation(content.toStringView(), ok)); return ok;
    default:
      break;
    }
    return false;
  }

  ENUM_AS_STRING_BEGIN(AsciiSignalFormatRule, Operation)
  ENUM_AS_STRING_DATA_5(Add, Assign, Divide, Multiply, Subtract);
  ENUM_AS_STRING_END

} // namespace GeopsyCore
