/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-13
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef MODELSET_H
#define MODELSET_H

#include <QtCore>
#include <QGpCoreMath.h>

#include "Index.h"

namespace DinverCore {

  class Model;
  class RealSpace;
  class ReportReader;

  class DINVERCORE_EXPORT ModelSet : private QSet<Model>, private IncreaseStorage
  {
    TRANSLATIONS("ModelSet")
  public:
    ModelSet(int parameterCount, int targetCount, int defaultCapacity=16384);
    ~ModelSet();

    bool importModelsFromAscii(RealSpace& parameterSpace, QString fileName);
    bool importModelsFromReport(RealSpace& parameterSpace, ReportReader& report, bool strict=true);
    bool importModels(RealSpace& parameterSpace, QString fileName, bool strict=true);
    SetIndex add(const RealSpace& parameterSpace, const double * misfit=nullptr);
    SetIndex add(Random& random);

    inline void setMisfit(const SetIndex& s, const TargetIndex& t, double m);
    void clear();

    void reserve(int n) {IncreaseStorage::reserve(n);}
    int size() const;
    int parameterCount() const {return _parameterCount;}
    int targetCount() const {return _targetCount;}

    const int * model(const SetIndex& i) const {return _parameters+_parameterCount*i.value();}
    const int * firstModel() const {return _parameters;}
    inline double misfit(const SetIndex& i, const TargetIndex& t) const;

    void lockForRead() {_modelLock.lockForRead();}
    void lockForWrite() {_modelLock.lockForWrite();}
    void unlock() {_modelLock.unlock();}

    void print(const int * model) const;
    void print(const SetIndex& i) const;
    SetIndex bestModel() const;
  private:
    void reallocate();

    int _parameterCount;
    int _targetCount;
    int * _parameters;
    double * _misfits;

    Mutex _setLock;
    ReadWriteLock _modelLock;
  };

  inline void ModelSet::setMisfit(const SetIndex& s, const TargetIndex& t, double m)
  {
    _misfits[_targetCount*s.value()+t.value()]=m;
  }

  inline double ModelSet::misfit(const SetIndex& i, const TargetIndex& t) const
  {
    return _misfits[_targetCount*i.value()+t.value()];
  }

} // namespace DinverCore

#endif // MODELSET_H
