/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2010-12-21
**  Copyright: 2010-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "Generator.h"
#include "ParentIndex.h"
#include "RealSpace.h"
#include "AbstractForward.h"
#include "VoronoiNavigator.h"
#include "ModelRepository.h"
#include "NewModel.h"

namespace DinverCore {

  /*!
    \class Generator Generator.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  Generator::Generator(ModelRepository * repository,
                       AbstractForward * forward,
                       int seed)
    : Thread(), _terminated(false)
  {
    TRACE;
    _type=MonteCarlo;
    _walkCount=2;

    // In a parallel implementations, random seed is meanningless
    // The random pathway depends upon CPU load and task distributions
    // Internal seed cannot be set from current time: all threads are created
    // at the same time, even if they will rapidly diverge, we ensure different values.
    _randomNumbers=new Random(seed);
    _repository=repository;
    _forward=forward;
    _generatorModelsSerialNumber=0; // First S/N starts at 1;
    _nav=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  Generator::~Generator()
  {
    TRACE;
    delete _randomNumbers;
    delete _forward;
    delete _nav;
  }

  QString Generator::type(Type t)
  {
    TRACE;
    switch(t) {
    case MonteCarlo:
      return "Monte-Carlo";
    case Neighborhood:
      break;
    case Borders:
      return "Borders";
    }
    return "Neighborhood";
  }

  /*!
  */
  void Generator::run()
  {
    TRACE;
    switch(_type) {
    case MonteCarlo:
      monteCarloLoop();
      break;
    case Neighborhood:
      neighborhoodLoop();
      break;
    case Borders:
      bordersLoop();
      break;
    }
  }

  /*!
  */
  void Generator::monteCarloLoop()
  {
    TRACE;

    while(!_terminated.value()) {
      NewModel * newModel=new NewModel;

      monteCarlo(newModel);
      if(newModel->isValidMisfit()) {
        _repository->addReport(newModel, _forward);
      }
      _repository->addModel(newModel);
    }
  }

  void Generator::monteCarlo(NewModel * newModel)
  {
    TRACE;

    RealSpace& parameterSpace=_forward->parameterSpace();
    int ndVar=parameterSpace.variableParameterCount();

    // Save current model is case of failure
    int * backupModel=new int[ndVar];
    for(int ip=0; ip<ndVar; ip++) {
      backupModel[ip]=parameterSpace.variableParameter(ip)->gridValue();
    }

    forever {
      if(_forward->randomModel(*_randomNumbers, _walkCount)) {
        if(misfit(newModel, parameterSpace)) {
          delete [] backupModel;
          return;
        }
      } else {
        // Current model does not satisfy all conditions, we fix it
        for(int ip=0; ip<ndVar; ip++) {
          parameterSpace.variableParameter(ip)->setGridValue(backupModel[ip]);
        }
        _forward->valueChanged();
        delete [] backupModel;
        _repository->stop(); // force termination of inversion
        return;
      }
    }
  }

  /*!
  */
  void Generator::neighborhoodLoop()
  {
    TRACE;

    while(!_terminated.value()) {
      // Selects randomly one of the best cells
      ParentIndex * parentIndex=_repository->createModel(_randomNumbers->ran2());
      if(!parentIndex) {
        break;
      }
      if(_generatorModelsSerialNumber!=parentIndex->serialNumber()) {
        delete _nav;
        _nav=new VoronoiNavigator(parentIndex->scaledModels());
        _generatorModelsSerialNumber=parentIndex->serialNumber();
      }
      NewModel * newModel=new NewModel;
      newModel->setParentIndex(parentIndex);

      neighborhood(newModel);
      if(newModel->isValidMisfit()) {
        _repository->addReport(newModel, _forward);
      }
      _repository->addModel(newModel);
    }
  }

  /*!
    Does random wolks inside the parent cell of \a newModel.
  */
  void Generator::neighborhood(NewModel * newModel)
  {
    TRACE;
    RealSpace& parameterSpace=_forward->parameterSpace();
    const ParentIndex * parentIndex=newModel->parentIndex();
    const ActiveIndex& activeIndex=parentIndex->activeIndex();
    int ndVar=parameterSpace.variableParameterCount();
    int nSteps=ndVar*_walkCount;

    _repository->modelToRealSpace(activeIndex, parameterSpace);
    _nav->setCurrentPoint(activeIndex);
    // Randomize starting axis
    // TODO: check the difference of classical incremental walk or random
    _nav->initDistances(ndVar-1);

    // Some parametrizations may need some updates before proceeding
    _forward->valueChanged();

    // Do random walks, if after 10 tries no new model can be generated (different from
    // all others generated so far), the cell is considered as atomic (only one point inside).
    int iStep, iTries;
    int xMin, xMax;
    ActiveIndex iMin, iMax;
    for(iTries=0; iTries<10; iTries++) {
      //printf("  Try %3i\n", iTries);
      int atomicAxis=0;
      // Estimation of the cell volume
      double parentVolume=1.0;
      for(iStep=0; iStep<nSteps; iStep++) {
        _nav->incrementAxis();
        //printf("    Step %3i Axis %i\n", iStep, _nav->currentAxis());
        //_nav->checkAxisDistances();
        Parameter * p=parameterSpace.variableParameter(_nav->currentAxis());
        p->getGridLimits(xMin, xMax);
        //printf("      Parameter min %5i max %5i\n", xMin, xMax);
        bool atomicBeforeCellLimits=(xMin>=xMax);
        _nav->cellLimits(xMin, xMax, iMin, iMax);
        if(atomicBeforeCellLimits) {
          ASSERT(xMin>=xMax);
        }

        // Update volume estimation
        parentVolume*=abs(xMax-xMin)+1;

        // Marks neighboring cells as active
        if(iMin.isValid()) {
          newModel->addNavigatorHit(iMin);
        }
        if(iMax.isValid()) {
          newModel->addNavigatorHit(iMax);
        }

        //printf("      Cell min %5i max %5i\n", xMin, xMax);
        // Check for atomic cell. A cell becomes atomic when it is atomic in all directions.
        if(xMin>=xMax) {
          atomicAxis++;
          if(atomicAxis==ndVar && iStep==(ndVar-1)) {
            return;  // validParent is false in newModel ==> parent will be set as atomic
          }
        } else {
          if(p->isFussy()) {     // Parameters tagged as fussy need a validation of the random value
            int nTries;
            // Allow for a maximum of 100 tries before giving up
            // This is not really a good solution but in some cases it avoids the process to be blocked
            for(nTries=0; nTries<100; nTries++) {
              p->setGridValue(_randomNumbers->uniform(xMin, xMax));
              // Some parametrizations may need some updates before proceeding
              _forward->valueChanged(p);
              if(_terminated.value()) {
                return;
              }
              if(_forward->isFussyOk(p)) {
                //if(nTries>0) {
                //  App::log(tr("Fussy parameter accepted after %1 tries\n").arg(nTries) );
                //}
                break;
              }
            }
            if(nTries==100) {
              //App::log(tr("Fussy parameter rejected after %1 tries, trying another model.\n").arg(nTries) );
              return;
            }
            _nav->setValue(p->gridValue());
          } else {
            int newValue=_randomNumbers->uniform(xMin, xMax);
            _nav->setValue(newValue);
            //printf("      New value %5i\n", newValue);
            p->setGridValue(newValue);
            // Some parametrizations may need some updates before proceeding
            _forward->valueChanged(p);
          }
        }
      }

      // ModelSet has its own mutex... thread safe.
      if(misfit(newModel, parameterSpace)) {
        // Average volume on all random walks, but half of it because of the newly the generated model
        // Children inherit parent volume
        newModel->setParentVolume(0.5*parentVolume/_walkCount);
        //printf("  --- ACCEPT new model ---\n" );
        //nav.currentPoint().print();
        // Check that generated model belongs to the right cell
        /*VectorList<int> iGenModel=nav.cellAt(nav.currentPoint());
        if( ! iGenModel.contains(nav.currentCell()) ) {
          printf("Model effectively NOT generated in cell %i but:\n", nav.currentCell());
          for(VectorList<int>::iterator it=iGenModel.begin(); it!=iGenModel.end(); it++) {
            printf("%i ", *it);
          }
          printf("\n" );
        }*/
        return;
      }
    }
    /*
       After several tries, it was not possible to generate a valid model that is
       not already in the model set. The are good chances that the current parent cell
       is close to an atomic cell.

       validParent is false in newModel ==> parent will be set as atomic
       (see ModelRepository::processQueue())
    */
    return;
  }

  /*!
  */
  void Generator::bordersLoop()
  {
    TRACE;
    _minimumMisfit=_repository->misfit(_repository->bestModelIndex());

    while(!_terminated.value()) {
      // Selects randomly one of the best cells
      ParentIndex * parentIndex=_repository->createModel(_randomNumbers->ran2());
      if(!parentIndex) {
        return;
      }
      if(_generatorModelsSerialNumber!=_repository->serialNumber()) { // If the same, keep iterating from the last point
        delete _nav;
        _nav=new VoronoiNavigator(parentIndex->scaledModels());
        _generatorModelsSerialNumber=parentIndex->serialNumber();
      }
      NewModel * newModel=new NewModel;
      newModel->setParentIndex(parentIndex);

      borders(newModel);
      if(newModel->isValidMisfit()) {
        _repository->addReport(newModel, _forward);
      }
      _repository->addModel(newModel);
    }
  }

  /*!
    Does random wolks inside the parent cell of \a newModel.
  */
  void Generator::borders(NewModel * newModel)
  {
    TRACE;
    RealSpace& parameterSpace=_forward->parameterSpace();
    const ParentIndex * parentIndex=newModel->parentIndex();
    const ActiveIndex& activeIndex=parentIndex->activeIndex();
    int ndVar=parameterSpace.variableParameterCount();
    int nSteps=ndVar*_walkCount;

    _repository->modelToRealSpace(activeIndex, parameterSpace);
    _forward->valueChanged();
    _nav->setCurrentPoint(activeIndex);
    _nav->initDistances(_randomNumbers->uniform(0, ndVar-1));
    //_nav->initDistances(ndVar-1);

    // Do random walks, if after 10 tries no new model can be generated (different from
    // all others generated so far), the cell is considered as atomic (only one point inside).
    int iStep, iTries;
    int xMin, xMax;
    SAFE_UNINITIALIZED(xMin, 0);
    SAFE_UNINITIALIZED(xMax, 0);
    ActiveIndex cellIndex;
    PdfCurve pdf;
    for(iTries=0; iTries<10; iTries++) {
      //printf("  Try %3i\n", iTries);
      for(iStep=0; iStep<nSteps; iStep++) {
        _nav->setCurrentAxis(_randomNumbers->uniform(0, ndVar-2));
        //_nav->incrementAxis();
        //printf("    Step %3i Axis %i\n", iStep, _nav->currentAxis());
        _nav->checkAxisDistances();
        Parameter * p=parameterSpace.variableParameter(_nav->currentAxis());
        p->getGridLimits(xMin, xMax);
        //PdfCurve pdfDebug=_nav->intersectionsDebug(xMin, xMax);
        pdf=_nav->intersections(xMin, xMax);
        /*int ipdf;
        if(!pdf.compare(pdfDebug, ipdf)) {
          _nav->setValue(pdfDebug.at(ipdf).x());
          _nav->setCurrentCell(pdfDebug.at(ipdf).cell());
          _nav->checkCurrentPoint(parameterSpace);
          _nav->setCurrentCell(pdf.at(ipdf).cell());
          _nav->checkCurrentPoint(parameterSpace);
        }*/
        pdf.setAcceptable(*parentIndex->activeModels(), _minimumMisfit);
        pdf.setBorders();
        //pdf.setStrictBorders();
        pdf.hits(newModel);

        if(p->isFussy()) {     // Parameters tagged as fussy need a validation of the random value
          int nTries;
          // Allow for a maximum of 100 tries before giving up
          // This is not really a good solution but in some cases it avoids the process to be blocked
          for(nTries=0; nTries<100; nTries++) {
            int val=pdf.generate(_randomNumbers->ran2(), cellIndex);
            if(!cellIndex.isValid()) {
              break;
            }
            p->setGridValue(val);
            // Some parametrizations may need some updates before proceeding
            _forward->valueChanged(p);
            if(_terminated.value()) {
              return;
            }
            if(_forward->isFussyOk(p)) {
              //if(nTries>0) {
              //  App::log(tr("Fussy parameter accepted after %1 tries\n").arg(nTries) );
              //}
              break;
            }
          }
          if(nTries==100) {
            //App::log(tr("Fussy parameter rejected after %1 tries, trying another model.\n").arg(nTries) );
            return;
          }
        } else {
          int val=pdf.generate(_randomNumbers->ran2(), cellIndex);
          if(cellIndex.isValid()) {
            p->setGridValue(val);
            _forward->valueChanged(p);  // Some parametrizations may need some updates before proceeding
          }
        }
        if(cellIndex.isValid()) {
          _nav->setValue(p->gridValue());
          _nav->setCurrentCell(cellIndex);
          ActiveIndex trueCell;
          if(!_nav->checkCurrentPoint(parameterSpace)) {
            pdf.debugPrint(parentIndex->activeModels());
            ASSERT(false);
          }
        } else { // Jumped outside the solution area, but close to border
                 //    Reset to original starting point
                 //    Compute misfit and add model
          _nav->setCurrentPoint(activeIndex);
          _nav->initDistances(_randomNumbers->uniform(0, ndVar-1));
          //_nav->initDistances(ndVar-1);
          break;
        }
        ASSERT(parameterSpace.isOkDebug());
      }

      //printf("Max steps %i\n", iStep);
      // ModelSet has its own mutex... thread safe.
      if(misfit(newModel, parameterSpace)) {
        return;
      }
      if(!cellIndex.isValid()) {  // Parameter space cannot be changed before misfit computation
        _repository->modelToRealSpace(activeIndex, parameterSpace);
        _forward->valueChanged();
      }
    }
    /*
       After several tries, it was not possible to generate a valid model that is
       not already in the model set. The are good chances that the current parent cell
       is close to an atomic cell.
    */
    return;
  }

  /*!
    If true is returned loop can be stopped: correct misfit found or end of inversion
  */
  bool Generator::misfit(NewModel * newModel, RealSpace& parameterSpace)
  {
    // Misfit is not directly set in the main repository.
    // With add, we just test if the model has already been calculated
    newModel->setIndex(_repository->allModels()->add(parameterSpace));
    if(newModel->index().isValid()) {
      bool ok=true;
      newModel->setMisfit(_forward->misfit(ok));
      newModel->setValidMisfit(ok);
      newModel->setValidParent(true);
      return true;
    } else {
      // Generated model already exist
      // For low dimension parameter spaces (<3), the space can be fully investigated
      // We put a threshold at 80% coverage to abort the inversion process.
      double nPossibleModels=parameterSpace.possibleCount();
      ASSERT(!newModel->isValidMisfit());
      _repository->allModels()->lockForRead();
      if(_repository->allModels()->size()>0.8*nPossibleModels) {
        _repository->allModels()->unlock();
        _repository->stop(); // force termination of inversion
        return true;
      } else {
        _repository->allModels()->unlock();
        return false; // Trying another one...
      }
    }
  }

} // namespace DinverCore
