/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-13
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ActiveModels.h"

namespace DinverCore {

  /*!
    \class ActiveModels ActiveModels.h
    \brief Brief description of class still missing

    Active model index remains valid during the whole life of this object.
    Exception is any call to clear() but this is not part of normal inversion
    run. Inversion is always stopped when clear() is called.
  */

  /*!
    Description of constructor still missing
  */
  ActiveModels::ActiveModels(ModelSet * allModels)
    : IncreaseStorage(1024)
  {
    _allModels=allModels;
    _infos=static_cast<ActiveModel *>(allocateVector(sizeof(ActiveModel)));
  }

  /*!
    Description of destructor still missing
  */
  ActiveModels::~ActiveModels()
  {
    TRACE;
    //printNavigatorHits();
    free(_infos);
  }

  void ActiveModels::reallocate()
  {
    _infos=static_cast<ActiveModel *>(reallocateVector((uchar *)_infos, sizeof(ActiveModel)));
  }

  /*!
    Used only when deleting all models from ensemble. Do not call this
    function while an inversion is running.
  */
  void ActiveModels::clear()
  {
    free(_infos);
    IncreaseStorage::clear();
    _infos=static_cast<ActiveModel *>(allocateVector(sizeof(ActiveModel)));
  }

  void ActiveModels::print(const ActiveIndex& i)
  {
    ActiveModel& m=_infos[i.value()];
    printf("Active index %5i vol %10.2lg : ", i.value(), m.volume());
    _allModels->print(m.index());
  }

  void ActiveModels::addNavigatorHits(const VectorList<ActiveIndex>& indexes)
  {
    int t=_allModels->size();
    for(VectorList<ActiveIndex>::const_iterator it=indexes.begin(); it!=indexes.end(); it++ ) {
      ActiveModel& m=at(*it);
      m.addNavigatorHit();
      m.setNavigatorHitTime(t);
    }
  }

  void ActiveModels::printNavigatorHits() const
  {
    ActiveIndex last;
    last.setValue(count());
    for(ActiveIndex i; i<last; i++) {
      const ActiveModel& m=at(i);
      int age=last.value()-m.navigatorHitTime();
      if(age<0) age=0;
      double temperature;
      if(age>0) {
        temperature=static_cast<double>(m.navigatorHitCount())/age;
      } else {
        temperature=1.0;
      }
      printf("ct= %i mi= %i nc= %i nt= %i age= %i temp= %lf\n",
             last.value(), i.value(),
             m.navigatorHitCount(), m.navigatorHitTime(), age, temperature);
    }
  }

  /*!
    Get the Manhatan distance between two models.
    Usually the number of bins is around 500, hence the risk of integer overflow is almost null.
  */
  int ActiveModels::distance(const ActiveIndex& i1, const ActiveIndex& i2) const
  {
    const int * m1=model(i1);
    const int * m2=model(i2);
    int d=0;
    for(int i=parameterCount()-1; i>=0; i--) {
      d+=abs(m1[i]-m2[i]);
    }
    return d;
  }

} // namespace DinverCore
