/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-09-06
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "WavefieldFunction.h"

namespace ArrayCore {

  /*!
    \class WavefieldFunction WavefieldFunction.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  WavefieldFunction::WavefieldFunction(int blockCount, const VectorList<Point2D>& sensors)
    : AbstractFunction()
  {
    _w=new Wavefield;
    _w->setBlockCount(blockCount);
    _w->setSensors(sensors);
    _w->setIncoherentNoise(1.0, 1.0);
  }

  /*!
    Description of destructor still missing
  */
  WavefieldFunction::~WavefieldFunction()
  {
    delete _w;
  }

  void WavefieldFunction::addObservedWave(double wavenumber, double theta,
                                          const WavefieldValues::Observations& obs)
  {
    _w->addWave(sqrt(obs.powerX), wavenumber, theta, obs.xiX);
    _obs.append(obs);
    _obs.last().init();
  }

  int WavefieldFunction::parameterSpaceDimension() const
  {
    return 2*_w->waveCount()+2;
  }

  void WavefieldFunction::setMaximumShift(Vector<double> maxShift) const
  {
    _w->setMaximumShift(maxShift);
  }

  void WavefieldFunction::setPrecision(Vector<double> prec) const
  {
    _w->setPrecision(prec);
  }

  void WavefieldFunction::setSteering(const FKSteering * s)
  {
    _w->initMisfit(_obs, s);
  }

  double WavefieldFunction::value(Vector<double>& p) const
  {
    _w->setParameters(p);
    _w->setFieldVectors();
    return _w->misfit(_obs);
  }

  void WavefieldFunction::gradient(const Vector<double>&, Vector<double>& grad) const
  {
    int n=grad.count()-2;
    for(int i=0; i<n; i++) {
      grad[i]=_w->misfitDerivative(i, _obs);
    }
    grad[n]=_w->misfitDerivative(n, _obs);
    grad[n+1]=_w->misfitDerivative(n+1, _obs);
  }

  void WavefieldFunction::showDetailedMisfit() const
  {
    if(App::verbosity()>=1) {
      _w->setFieldVectors();
      App::log(_w->detailedMisfit(_obs));
    }
  }

  void WavefieldFunction::showResults(double minValue, int iterationCount, double frequency) const
  {
    App::log(tr("====================================================\n"
                "  Minimum misfit %1 found after %2 iterations\n")
             .arg(minValue)
             .arg(iterationCount));
    int nw=_w->waveCount();
    for(int iw=0; iw<nw; iw++) {
      App::log(tr("    Wave %1: v %2 az %3 a %4 xi %5 deg\n")
               .arg(iw)
               .arg(2.0*M_PI*frequency/_w->wavenumber(iw), 20, 'g')
               .arg(Angle::radiansToDegrees(Angle::mathToGeographic(_w->theta(iw))), 20, 'g')
               .arg(_w->amplitude(iw), 20, 'g')
               .arg(Angle::radiansToDegrees(_w->ellipticity(iw)), 20, 'g'));
    }
    App::log(tr("  Noise R %1 sigma %2\n\n")
             .arg(_w->incoherentNoiseRation())
             .arg(_w->sigma()));
  }

  double WavefieldFunction::admissibleStep(const Vector<double>& x, const Vector<double>& step) const
  {
    double alpha, maxAlpha=std::numeric_limits<double>::infinity();
    int param=-1;
    int n=x.count()-2;
    for(int i=0; i<n; i+=2) {
      alpha=(0.001-x[i])/step[i];
      if(alpha>0.0 && alpha<maxAlpha) {
        maxAlpha=alpha;
        param=i;
      }
      alpha=fabs(0.1*x[i]/step[i]);
      if(alpha<maxAlpha) {
        maxAlpha=alpha;
        param=i;
      }

      alpha=fabs(M_PI/90.0/step[i+1]);
      if(alpha<maxAlpha) {
        maxAlpha=alpha;
        param=i+1;
      }
    }

    alpha=(0.001-x[n])/step[n];
    if(alpha>0.0 && alpha<maxAlpha) {
      maxAlpha=alpha;
      param=n;
    }
    alpha=fabs(0.1*x[n]/step[n]);
    if(alpha<maxAlpha) {
      maxAlpha=alpha;
      param=n;
    }

    alpha=(0.001-x[n+1])/step[n+1];
    if(alpha>0.0 && alpha<maxAlpha) {
      maxAlpha=alpha;
      param=n+1;
    }
    alpha=fabs(0.1*x[n+1]/step[n+1]);
    if(alpha<maxAlpha) {
      maxAlpha=alpha;
      param=n+1;
    }

    printf("maximum alpha %lf (param %i %lf %lf)\n", maxAlpha, param, x[param], step[param]);
    return maxAlpha;
  }

  void WavefieldFunction::checkDerivatives()
  {
    PrivateVector<double> p0(parameterSpaceDimension());
    PrivateVector<double> p(parameterSpaceDimension());
    _w->getParameters(p0);
    for(int i=0; i<parameterSpaceDimension(); i++) {
      p.copyValues(p0);
      double pimin, pimax, dpi;
      if(i%2==1 && i<parameterSpaceDimension()-2) {
        pimin=-M_PI/2.0;
        pimax=M_PI/2.0;
        dpi=M_PI/180.0;
      } else {
        pimin=0.1*p[i];
        pimax=10.0*p[i];
        dpi=0.05*p[i];
      }
      QFile f("/tmp/wavefield_p"+QString::number(i));
      f.open(QIODevice::WriteOnly);
      QTextStream s(&f);
      App::log(tr("Parameter %1 from %2 to %3\n").arg(i).arg(pimin).arg(pimax));
      for(p[i]=pimin; p[i]<pimax; p[i]+=dpi) {
        _w->setParameters(p);
        _w->setFieldVectors();
        double m=_w->misfit(_obs);
        double dm=_w->misfitDerivative(i, _obs);
        s << p[i] << " " << m << " " << dm << "\n";
      }
      f.close();
    }
    _w->setParameters(p0);
  }

  bool WavefieldFunction::saveObservations(const QString& fileName) const
  {
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      s << _obs.count() << " " << _w->blockCount() << Qt::endl;
      for(int i=0; i<_obs.count(); i++) {
        const WavefieldValues::Observations& o=_obs[i];
        s << QString::number(_w->wavenumber(i), 'g', 20) << " "
          << QString::number(_w->theta(i), 'g', 20) << Qt::endl
          << QString::number(o.xiH, 'g', 20) << " "
          << QString::number(o.powerH, 'g', 20) << " "
          << QString::number(o.kConcavityH, 'g', 20) << " "
          << QString::number(o.xiConcavityH, 'g', 20) << Qt::endl
          << QString::number(o.xiZ, 'g', 20) << " "
          << QString::number(o.powerZ, 'g', 20) << " "
          << QString::number(o.kConcavityZ, 'g', 20) << " "
          << QString::number(o.xiConcavityZ, 'g', 20) << Qt::endl
          << QString::number(o.xiX, 'g', 20) << " "
          << QString::number(o.powerX, 'g', 20) << " "
          << QString::number(o.kConcavityX, 'g', 20) << " "
          << QString::number(o.xiConcavityX, 'g', 20) << Qt::endl;
      }
      return true;
    } else {
      App::log(tr("Cannot write to file '%1'\n").arg(fileName));
      return false;
    }
  }

  bool WavefieldFunction::loadObservations(const QString& fileName)
  {
    QFile f(fileName);
    if(f.open(QIODevice::ReadOnly)) {
      QTextStream s(&f);
      bool ok=true;
      LineParser lp;
      QString l=s.readLine();
      lp.setString(l);
      int nw=lp.toInt(0, ok);
      _w->setBlockCount(lp.toInt(1, ok));
      _w->clearWaves();
      _obs.clear();
      if(nw>0) {
        WavefieldValues::Observations o;
        for(int i=0; i<nw; i++) {
          l=s.readLine();
          lp.setString(l);
          double k=lp.toDouble(0, ok);
          double theta=lp.toDouble(1, ok);
          l=s.readLine();
          lp.setString(l);
          o.xiH=lp.toDouble(0, ok);
          o.powerH=lp.toDouble(1, ok);
          o.kConcavityH=lp.toDouble(2, ok);
          o.xiConcavityH=lp.toDouble(3, ok);
          l=s.readLine();
          lp.setString(l);
          o.xiZ=lp.toDouble(0, ok);
          o.powerZ=lp.toDouble(1, ok);
          o.kConcavityZ=lp.toDouble(2, ok);
          o.xiConcavityZ=lp.toDouble(3, ok);
          l=s.readLine();
          lp.setString(l);
          o.xiX=lp.toDouble(0, ok);
          o.powerX=lp.toDouble(1, ok);
          o.kConcavityX=lp.toDouble(2, ok);
          o.xiConcavityX=lp.toDouble(3, ok);
          if(!ok) {
            App::log(tr("Error parsing observation %1\n").arg(i));
            break;
          }
          _obs.append(o);
          _w->addWave(o.powerX, k, theta, o.xiX);
        }
      }
      return ok;
    } else {
      App::log(tr("Cannot write to file '%1'\n").arg(fileName));
      return false;
    }
  }
} // namespace ArrayCore

