/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-08
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef STEERINGFACTORS_H
#define STEERINGFACTORS_H

#include "ArrayCoreDLLExport.h"
#include "ArraySelection.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT SteeringFactors
  {
  public:
    SteeringFactors(int count, double kmax, const ArraySelection * array);
    ~SteeringFactors();

    void setGrid();

    class Iterator
    {
    public:
      inline Iterator(const SteeringFactors * fac);

      inline void setK(int kx, int ky);

      int xHighIndex() const {return _xHighIndex;}
      int xLowIndex() const {return _xLowIndex;}
      int yHighIndex() const {return _yHighIndex;}
      int yLowIndex() const {return _yLowIndex;}
    private:
      int _shift, _mask, _stationCount;
      int _xHighIndex, _xLowIndex, _yHighIndex, _yLowIndex;
    };

    inline Complex value(const Iterator& it, int station) const;
    inline void oneComponent(const Iterator& it, Complex * e) const;

    static bool test(int count, double kmax, const ArraySelection * array);

    void print();
  private:
    friend class Iterator;
    int _stationCount, _count;
    int _shift, _mask;
    int _lowCount, _highCount;
    int _lowHalfCount, _highHalfCount;

    Complex * _highX, * _lowX, * _highY, * _lowY;
  };

  inline SteeringFactors::Iterator::Iterator(const SteeringFactors * fac)
  {
    _shift=fac->_shift;
    _mask=fac->_mask;
    _stationCount=fac->_stationCount;
  }

  inline void SteeringFactors::Iterator::setK(int kx, int ky)
  {
    if(kx<0) {
      kx=-kx;
      _xHighIndex=-(kx >> _shift)*_stationCount;
      _xLowIndex=-(kx & _mask)*_stationCount;
    } else {
      _xHighIndex=(kx >> _shift)*_stationCount;
      _xLowIndex=(kx & _mask)*_stationCount;
    }
    if(ky<0) {
      ky=-ky;
      _yHighIndex=-(ky >> _shift)*_stationCount;
      _yLowIndex=-(ky & _mask)*_stationCount;
    } else {
      _yHighIndex=(ky >> _shift)*_stationCount;
      _yLowIndex=(ky & _mask)*_stationCount;
    }
  }

  inline Complex SteeringFactors::value(const Iterator& it, int station) const
  {
    Complex c;
    c=_highX[it.xHighIndex()+station];
    c*=_lowX[it.xLowIndex()+station];
    c*=_highY[it.yHighIndex()+station];
    c*=_lowY[it.yLowIndex()+station];
    return c;
  }

  inline void SteeringFactors::oneComponent(const Iterator& it, Complex * e) const
  {
    Complex * data;
    data=_highX+it.xHighIndex();
    for(int i=0; i<_stationCount; i++) {
      e[i]=data[i];
    }
    data=_lowX+it.xLowIndex();
    for(int i=0; i<_stationCount; i++) {
      e[i]*=data[i];
    }
    data=_highY+it.yHighIndex();
    for(int i=0; i<_stationCount; i++) {
      e[i]*=data[i];
    }
    data=_lowY+it.yLowIndex();
    for(int i=0; i<_stationCount; i++) {
      e[i]*=data[i];
    }
  }

} // namespace ArrayCore

#endif // STEERINGFACTORS_H

