/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-08
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SteeringFactors.h"

namespace ArrayCore {

  /*!
    \class SteeringFactors SteeringFactors.h
    \brief Brief description of class still missing

    Efficiency of L1-L2-L3 cache usage is linked to the size of this caching object.
    Interesting online book (chapter 22 about cache basics):
    http://www.cs.umd.edu/~meesh/411/CA-online

    L1 on Core i7 (2010) is 32 kB, L2 is 256 kB and L3 is 6Mb.

    Each of the four vectors (x/y, low/high) for 14 stations and 123 wavenumber values (-61 to +61)
    weights 3.5 kB. The cross-spectral matrix weights also 3.5 kB. Steering vector weights
    240 B. Hence the total memory required to scan the FK map is only less than 18 kB.
    Hence several cross-spectral matrices can be available in cache L1 at the same time.
    The steering factors will be more used and are likely to reside longer in the cache.

    The quantity of operations to compose the 2D steering vector is probably too high
    compared to the time gained by limiting memory access to cache L1. There is maybe
    something to gain for the refined search (still a factor 2 observed compared to plain
    compplex exp computation).

    Tested for 14 (3 Mb) station array and 38 (236 Mb de cache): 10% and 40% longer, respectively.
  */

  /*!
    Produced values are from \a -kmax to \a kmax with 2*\a count+1 steps.
  */
  SteeringFactors::SteeringFactors(int count, double kmax, const ArraySelection * array)
  {
    _stationCount=array->count();
    _count=count;

    int base=Number::nextPowerOfTwo(qCeil(sqrt(count)));
    _mask=base-1;
    _shift=0;
    for(int i=1; i<=_mask; i=i << 1, _shift++) {}

    _lowHalfCount=_mask;
    _highHalfCount=((count & ~_mask) >> _shift);
    _lowCount=_lowHalfCount*2+1;
    _highCount=_highHalfCount*2+1;

    // Pointers to index=0 (indexed from -count to +count)
    int n;
    n=_lowCount*_stationCount;
    _lowX=new Complex[n]+_lowHalfCount*_stationCount;
    _lowY=new Complex[n]+_lowHalfCount*_stationCount;
    n=_highCount*_stationCount;
    _highX=new Complex[n]+_highHalfCount*_stationCount;
    _highY=new Complex[n]+_highHalfCount*_stationCount;

    double dk=kmax/static_cast<double>(count);

    double kf;
    int i0;
    for(int k=-_lowHalfCount; k<=-1; k++) {
      kf=-static_cast<double>(k)*dk;
      i0=k*_stationCount;
      for(int i=0; i<_stationCount; i++) {
        const Point2D& r=array->relativePos(i);
        _lowX[i0+i].setUnitExp(kf*r.x());
        _lowY[i0+i].setUnitExp(kf*r.y());
      }
    }
    for(int i=0; i<_stationCount; i++) {
      _lowX[i]=1.0;
      _lowY[i]=1.0;
    }
    for(int k=1; k<=_lowHalfCount; k++) {
      kf=-static_cast<double>(k)*dk;
      i0=k*_stationCount;
      for(int i=0; i<_stationCount; i++) {
        const Point2D& r=array->relativePos(i);
        _lowX[i0+i].setUnitExp(kf*r.x());
        _lowY[i0+i].setUnitExp(kf*r.y());
      }
    }

    for(int k=-_highHalfCount; k<=-1; k++) {
      kf=-static_cast<double>(k*base)*dk;
      i0=k*_stationCount;
      for(int i=0; i<_stationCount; i++) {
        const Point2D& r=array->relativePos(i);
        _highX[i0+i].setUnitExp(kf*r.x());
        _highY[i0+i].setUnitExp(kf*r.y());
      }
    }
    for(int i=0; i<_stationCount; i++) {
      _highX[i]=1.0;
      _highY[i]=1.0;
    }
    for(int k=1; k<=_highHalfCount; k++) {
      kf=-static_cast<double>(k*base)*dk;
      i0=k*_stationCount;
      for(int i=0; i<_stationCount; i++) {
        const Point2D& r=array->relativePos(i);
        _highX[i0+i].setUnitExp(kf*r.x());
        _highY[i0+i].setUnitExp(kf*r.y());
      }
    }
  }

  /*!
    Description of destructor still missing
  */
  SteeringFactors::~SteeringFactors()
  {
    delete [] (_lowX-_lowHalfCount*_stationCount);
    delete [] (_highX-_highHalfCount*_stationCount);
    delete [] (_lowY-_lowHalfCount*_stationCount);
    delete [] (_highY-_highHalfCount*_stationCount);
  }

  void SteeringFactors::print()
  {
    printf("  Base  %i\n"
           "  Shift %i\n"
           "  Mask  %i\n"
           "  Low count  %i\n"
           "  High count %i\n", _mask+1, _shift, _mask, _lowCount, _highCount);
  }

  bool SteeringFactors::test(int count, double kmax, const ArraySelection * array)
  {
    SteeringFactors f(count, kmax, array);
    f.print();
    double dk=kmax/static_cast<double>(count);
    Complex c1, c2;
    double tolerance=1e-14;
    Iterator it(&f);
    for(int kx=-count; kx<=count; kx++) {
      for(int ky=-count; ky<=count; ky++) {
        it.setK(kx, ky);
        ComplexMatrix e;
        f.oneComponent(it, e.values());
        for(int i=array->count()-1; i>=0; i--) {
          const Point2D& r=array->relativePos(i);
          Point2D k(kx*dk, ky*dk);
          c1.setUnitExp(-r.scalarProduct(k));
          c2=f.value(it, i);
          if(fabs(c1.re()-c2.re())>tolerance ||
             fabs(c1.im()-c2.im())>tolerance) {
            App::log(tr("Test failed for count=%1 at k=%2 and station %3:\n"
                        "   %4 instead of %5\n")
                     .arg(count)
                     .arg(k.toString())
                     .arg(i)
                     .arg(c2.toString())
                     .arg(c1.toString()));
            return false;
          }
          if(c2!=e.at(i, 0)) {
            App::log(tr("Steering vector not consistent with value for station %1\n")
                     .arg(i));
            return false;
          }
        }
      }
    }
    App::log(tr("Test successful for count=%1\n").arg(count));
    return true;
  }

} // namespace ArrayCore

