/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-10-25
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SensorOrientationFunction.h"
#include "CrossSpectrumSplit.h"

namespace ArrayCore {

  /*!
    \class SensorOrientationFunction SensorOrientationFunction.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SensorOrientationFunction::SensorOrientationFunction(FKCache * cache)
    : ConvFKVertical(cache)
  {
    TRACE;
    _crossSpectra=nullptr;
    _stationCount=0;
    _innerStations=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  SensorOrientationFunction::~SensorOrientationFunction()
  {
    TRACE;
    qDeleteAll(_powers);
  }

  void SensorOrientationFunction::setCrossSpectra(const VectorList<CrossSpectrumSplit *> *cs)
  {
    _crossSpectra=cs;
    _stationCount=cs->first()->stationCount();

    int totalWavenumberCount=0;
    VectorList<CrossSpectrumSplit *>::const_iterator it;
    for(it=_crossSpectra->begin(); it!=_crossSpectra->end(); it++) {
      totalWavenumberCount+=(*it)->wavenumberCount();
    }
    _powers.resize(totalWavenumberCount);
    for(int i=totalWavenumberCount-1; i>=0; i--) {
      _powers[i]=new FKPower(cache()->array()->count(), 3);
    }

    size_t n=3*_stationCount;

    _psiE.resize(n);
    _psiN.resize(n);
    _dpsiE.resize(n);
    _dpsiN.resize(n);

    _psiE.zero();
    _psiN.zero();
    _dpsiE.zero();
    _dpsiN.zero();
  }

  double SensorOrientationFunction::value(Vector<double>& x) const
  {
    // Build the Psi matrices
    int n=_stationCount;
    int n2=2*n;
    int in;
    for(int i=0; i<n; i++) {
      Complex& vE=_psiE.at(i, i);
      Complex& vN=_psiN.at(i, i);
      const double& vX=x[i+1];
      vE=cos(vX);
      vN=sin(vX);
      in=i+n;
      _psiE.at(in, in)=vE;
      _psiN.at(in, in)=vN;
      in=i+n2;
      _psiE.at(in, in)=vE;
      _psiN.at(in, in)=vN;
    }
    _ell.setUnitExp(x[0]);

    PrivateVector<double> k(2, 0.0);
    VectorList<CrossSpectrumSplit *>::const_iterator it;
    double sum=0.0;
    int powerIndex=0;
    for(it=_crossSpectra->begin(); it!=_crossSpectra->end(); it++) {
      const CrossSpectrumSplit& cs=**it;
      // Build the corrected cross-spectrum matrix and invert it
      _finv=_psiE*cs.fEZ()+_psiN*cs.fZE();
      _finv+=_finv.conjugate().transposed();
      _finv+=_psiE*(cs.fEE()*_psiE+cs.fEN()*_psiN);
      _finv+=_psiN*(cs.fNE()*_psiE+cs.fNN()*_psiN);
      _finv+=cs.fZZ();
      //_psiE.toOctaveFile("/tmp/octave.m", "psiE");
      //_psiN.toOctaveFile("/tmp/octave.m", "psiN");
      //_finv.toOctaveFile("/tmp/octave.m", "f");
      _finv.choleskyInvert();
      //_finv.toOctaveFile("/tmp/octave.m", "finv");
      for(int i=cs.wavenumberCount()-1; i>=0; i--) {
        k[0]=cs.kx(i);
        k[1]=cs.ky(i);
        FKPower * powerEngine=_powers.at(powerIndex++);
        steering()->threeComponentRayleigh(k, powerEngine);
        const Vector<Complex>& e=FKSteering::threeComponentRayleighCross(_ell, powerEngine);
        //e.toOctaveFile("/tmp/octave.m", "e");
        powerEngine->setCrossSpectrum(&_finv);
        powerEngine->setHighResolutionValue(e);
        sum+=powerEngine->value();
      }
    }
    return -sum;
  }

  void SensorOrientationFunction::gradient(const Vector<double>&, Vector<double>& grad) const
  {
    int n=_stationCount;
    int n2=2*n;

    // Reset gradient
    for(int i=0; i<=n; i++) {
      grad[i]=0.0;
    }
    VectorList<CrossSpectrumSplit *>::const_iterator it;
    int powerIndex=0;
    for(it=_crossSpectra->begin(); it!=_crossSpectra->end(); it++) {
      const CrossSpectrumSplit& cs=**it;
      for(int i=cs.wavenumberCount()-1; i>=0; i--) {
        FKPower * powerEngine=_powers.at(powerIndex++);
        double value2=powerEngine->value()*powerEngine->value();
        const Vector<Complex>& de=FKSteering::threeComponentRayleighCrossGradient(_ell, powerEngine);
        grad[0]+=-2.0*value2*powerEngine->ellipticityGradient(de);
        for(int i=0; i<n; i++) {
          if(!_innerStations[i]) {
            continue;
          }
          // Init dpsi
          Complex& dpsiE=_dpsiE.at(i, i);
          Complex& dpsiN=_dpsiN.at(i, i);
          dpsiE=-_psiN.at(i, i);
          dpsiN=_psiE.at(i, i);
          int in=i+n;
          _dpsiE.at(in, in)=dpsiE;
          _dpsiN.at(in, in)=dpsiN;
          in=i+n2;
          _dpsiE.at(in, in)=dpsiE;
          _dpsiN.at(in, in)=dpsiN;

          ComplexMatrix df;
          // dF/dpsi_i
          df=_dpsiE*cs.fEZ()+_dpsiN*cs.fZE();
          df+=df.conjugate().transposed();
          df+=_dpsiE*(cs.fEE()*_psiE+cs.fEN()*_psiN);
          df+=_dpsiN*(cs.fNE()*_psiE+cs.fNN()*_psiN);
          df+=_psiE*(cs.fEE()*_dpsiE+cs.fEN()*_dpsiN);
          df+=_psiN*(cs.fNE()*_dpsiE+cs.fNN()*_dpsiN);
          /*_psiE.toOctaveFile("/tmp/octave.m", "psiE");
          _psiN.toOctaveFile("/tmp/octave.m", "psiN");
          _dpsiE.toOctaveFile("/tmp/octave.m", "dpsiE");
          _dpsiN.toOctaveFile("/tmp/octave.m", "dpsiN");
          _fEZ.toOctaveFile("/tmp/octave.m", "fEZ");
          _fZE.toOctaveFile("/tmp/octave.m", "fZE");
          _fEE.toOctaveFile("/tmp/octave.m", "fEE");
          _fEN.toOctaveFile("/tmp/octave.m", "fEN");
          _fNE.toOctaveFile("/tmp/octave.m", "fNE");
          _fNN.toOctaveFile("/tmp/octave.m", "fNN");
          df.toOctaveFile("/tmp/octave.m", "df");*/
          grad[i+1]+=value2*powerEngine->orientationGradient(df);

          // Leave dpsi as zero
          dpsiE=0.0;
          dpsiN=0.0;
          in=i+n;
          _dpsiE.at(in, in)=0.0;
          _dpsiN.at(in, in)=0.0;
          in=i+n2;
          _dpsiE.at(in, in)=0.0;
          _dpsiN.at(in, in)=0.0;
        }
      }
    }
    grad.changeSign();
  }

  void SensorOrientationFunction::scan2D(double range, int axis1, int axis2, const QString& fileName)
  {
    App::log(tr("2D scanning along axis %1 and %2\n").arg(axis1).arg(axis2));
    QFile f(fileName+QString("_%1_%2").arg(axis1, 2, 10, QChar('0')).arg(axis2, 2, 10, QChar('0')));
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);
    s << "x y val\n";
    double delta=range/15.0;
    PrivateVector<double> x(_stationCount+1, 0.0);
    x[0]=Angle::degreesToRadians(60.9152170840991); // 3Hz
    x[1]=Angle::degreesToRadians(-2.57207);
    x[2]=Angle::degreesToRadians(-0.852483);
    x[3]=Angle::degreesToRadians(3.62209);
    x[4]=Angle::degreesToRadians(-1.13666);
    x[5]=Angle::degreesToRadians(-1.13606);
    x[6]=Angle::degreesToRadians(0.86436);
    x[7]=Angle::degreesToRadians(4.25015);
    x[8]=Angle::degreesToRadians(2.90155);
    x[9]=Angle::degreesToRadians(3.89066);
    x[10]=Angle::degreesToRadians(-2.41819);
    x[11]=Angle::degreesToRadians(-3.0344);
    x[12]=Angle::degreesToRadians(-0.497615);
    x[13]=Angle::degreesToRadians(3.61134);
    x[14]=Angle::degreesToRadians(3.0975);
    for(double x1=-range; x1<=range; x1+=delta) {
      x[axis1]+=x1;
      for(double x2=-range; x2<=range; x2+=delta) {
        x[axis2]+=x2;
        s << Angle::radiansToDegrees(x1) << " " << Angle::radiansToDegrees(x2) << " "
          << value(x) << "\n";
        x[axis2]-=x2;
      }
      x[axis1]-=x1;
    }
    f.close();
  }

} // namespace ArrayCore

