/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-11-29
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "FKWorker.h"
#include "FKTaskManager.h"
#include "ArrayStations.h"
#include "WaveNumberConverter.h"
#include "FKResults.h"

//#define INSPECT_FREQUENCY 5.9958019753561773157
//#define INSPECT_TIME "20000101000000.000000"

// Run with verbosity level to 3 to get the report.
// Level 3 also activate pre-loading of signals in AbstractTool.
// Configure with '-D PROCESS_TIMING'

namespace ArrayCore {

#ifdef PROCESS_TIMING
  static double ttmg=0.0;
  static Mutex globalTimeMutex;
#endif

  /*!
    \class FKWorker FKWorker.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  FKWorker::FKWorker(const ArraySelection * array,
                     const FKParameters * param)
  {
    TRACE;
    _crossSpectrum=new FKCrossSpectrum(array, param);
    _peaks.setTimeReference(array->array()->first()->minTime());
    _search=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  FKWorker::~FKWorker()
  {
    TRACE;
    delete _search;
#ifdef PROCESS_TIMING
    App::log(3, tr("Merging results: %1 (ms)\n")
             .arg(ttmg*1e-6));
#endif
  }

  bool FKWorker::setGrid(FKCache * cache)
  {
    TRACE;
    AbstractFKFunction * f=AbstractFKFunction::create(taskManager()->array().array()->mode(),
                                                      crossSpectrum()->parameters(), cache);
    if(f) {
      f->setCrossSpectrum(crossSpectrum());
      const FKParameters * param=crossSpectrum()->parameters();
      // Initialization that require the virtual mechanism (e.g. allocation of power engine)
      f->initialize(*param);
      // Limiting search to kmin speedup computation but may pick alias peaks
      // Better to filter out after FK search
      //f->setMinimumWavenumber(param->minimumWaveNumber());
      // Reduce the gridded box to a circle of radius effective grid size
      f->setMaximumWavenumber(param->effectiveGridSize());
      f->setMaximumSlowness(param->maximumSlowness());
      _search=f->createSearch(*param);
      f->setGridStep(param->gridStep());
      return true;
    } else {
      delete _search;
      _search=nullptr;
      return false;
    }
  }

  void FKWorker::process(const VectorList<int>& blocks)
  {
    WaveNumberConverter conv(_crossSpectrum->timeWindows()->frequency().center());
    const TimeWindowList& winList=_crossSpectrum->timeWindows()->list();
    conv.setTime(winList.at(blocks.first()).start());
    APP_LOG(3, tr("Process at time %1 and at %2 Hz\n")
            .arg(conv.time().toString(DateTime::defaultFormat))
            .arg(conv.frequency()))

#ifdef INSPECT_FREQUENCY
    if(fabs(conv.frequency()-INSPECT_FREQUENCY)<0.001) {
      App::log(tr("Inspect selected frequency %1 Hz\n").arg(conv.frequency()));
    } else {
      return;
    }
#endif
#ifdef INSPECT_TIME
    DateTime t;
    t.fromString(INSPECT_TIME);
    if(fabs(conv.time().secondsTo(t))<0.1) {
      App::log(tr("Inspect time window starting at %1\n").arg(conv.time().toString()));
    } else {
      return;
    }
#endif
    if(!setSlownessRange()) {
      return;
    }
    const FKParameters * param=crossSpectrum()->parameters();
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_search->function());
    function->setTotalLimits(crossSpectrum()->timeWindows()->frequency().squaredOmega());
    function->setAzimuthRange(param->minimumAzimuth(),
                              param->maximumAzimuth());

    if(crossSpectrum()->calculate(blocks, function)) {
      if(param->brightSpots()<=0) {
        localizeMax();
        addPeaks(conv);
      } else {
        decompose(conv);
      }
    }
  }

  void FKWorker::decompose(const WaveNumberConverter& conv)
  {
    const FKParameters * param=crossSpectrum()->parameters();
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_search->function());
    int brightSpots=0;
    double power0=0.0;
    while(true) {
      _search->globalMax(param->absoluteThreshold());
      if(_search->maximaCount()!=1) {
        break;
      }
      const FunctionSearchMaximum& m=**_search->begin();
      if(addPeak(m, conv, param, function)) {
        FKPeaks::Value * value=_peaks.lastValue();
        if(power0==0.0) {
          power0=value->power;
        } else {
          if(value->power<power0*param->relativeThreshold()) {
            _peaks.removeLast();
            break;
          }
        }

        if(brightSpots>=param->brightSpots()) {
          break;
        }
        if(!function->remove(m.position(), *value)) {
          // End of the story, the modified matrix cannot be inverted
          return;
        }
        brightSpots++;
      }
    }
  }

  void FKWorker::mergeResults()
  {
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tmg;
    chrono.start();
#endif
    FKResults * results=static_cast<FKResults *>(taskManager()->results());
    if(results) {
      App::log(tr("Merging results for worker %1...\n").arg(objectName()));
      results->add(_peaks);
    }
#ifdef PROCESS_TIMING
      tmg=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      ttmg+=tmg;
      globalTimeMutex.unlock();
#endif
  }

  AbstractFKFunction * FKWorker::function()
  {
    if(setSlownessRange()) {
      AbstractFKFunction * f=static_cast<AbstractFKFunction *>(_search->function());
      f->setTotalLimits(crossSpectrum()->timeWindows()->frequency().squaredOmega());
      f->setAzimuthRange(crossSpectrum()->parameters()->minimumAzimuth(),
                         crossSpectrum()->parameters()->maximumAzimuth());
      return f;
    } else {
      return nullptr;
    }
  }

  void FKWorker::localizeMax()
  {
    TRACE;
    const FKParameters * param=crossSpectrum()->parameters();
    int nMax=param->maximumPeakCount(crossSpectrum()->array().count());
    if(nMax>1) {
      _search->localMax(nMax,
                        param->absoluteThreshold(),
                        param->relativeThreshold());
    } else {
      _search->globalMax(param->absoluteThreshold());
    }
  }

  bool FKWorker::addPeaks(const WaveNumberConverter& conv)
  {
    TRACE;
    const FKParameters * param=static_cast<const FKParameters *>(taskManager()->parameters());
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_search->function());
    FunctionMaximaIterator it;
    APP_LOG(4, tr("    Found %1 peaks\n").arg(_search->maximaCount()))
    for(it=_search->begin(); it!=_search->end(); ++it) {
      addPeak(**it, conv, param, function);
    }
    return true;
  }

  bool FKWorker::addPeak(const FunctionSearchMaximum& m, const WaveNumberConverter& conv,
                         const FKParameters * param, AbstractFKFunction * function)
  {
    const Vector<double>& k=m.position();
    double slowness=conv.slowness(k);
    if(slowness>param->minimumSlowness() &&
       !function->isOnLimit(k, conv.frequency())) {
      FKPeaks::Value * val=_peaks.add();
      if(val) {
        val->time=_peaks.time(conv.time());
        val->frequency=conv.frequency();
        val->slowness=slowness;
        val->azimuth=conv.azimuth(k);
        val->valid=true;
        val->extraValues=nullptr;
        function->setAssociatedResults(k, m.value(), *val);
        APP_LOG(4, tr("%1 Hz: new peak at %2 m/s propagating towards %3 with ellippticity %4 deg\n")
                            .arg(conv.frequency())
                            .arg(1.0/slowness)
                            .arg(conv.azimuth(k))
                            .arg(Angle::radiansToDegrees(::atan(val->ellipticity))))
      } else {
        return false;
      }
      return true;
    }
    return false;
  }

  bool FKWorker::setSlownessRange()
  {
    TRACE;
    const FKParameters * param=crossSpectrum()->parameters();
    AbstractFKFunction * function=static_cast<AbstractFKFunction *>(_search->function());
    double f=crossSpectrum()->timeWindows()->frequency().center();
    if(param->isRefine()) {
      //App::log(tr("Looking for frequency %1 in refinement limits\n").arg(f)
      //              << param->refineMinimumSlowness().toString() << Qt::endl;
      int index=param->refineMinimumSlowness().indexOf(f, 1e-4); // default frequency step is 2.5%
      if(index<0) {
        APP_LOG(1, tr("  Skipped frequency %1 Hz\n").arg(f))
        return false;
      } else {
        function->setMinimumSlowness(param->refineMinimumSlowness().y(index));
        function->setMaximumSlowness(param->refineMaximumSlowness().y(index));
        APP_LOG(1, tr("  Minimum slowness %1 s/m at %2 Hz\n").arg(function->minimumSlowness()).arg(f))
        APP_LOG(1, tr("  Maximum slowness %1 s/m at %2 Hz\n").arg(function->maximumSlowness()).arg(f))
        return true;
      }
    } else {
      // Rayleigh and Love must be computed on the same frequency range but
      // results may accept values on distinct frequency ranges.
      if(param->acceptFrequency(f)) {
        return true;
      } else {
        APP_LOG(1, tr("  Skipped frequency %1 Hz\n").arg(f))
        return false;
      }
    }
  }

} // namespace ArrayCore

