/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-04-04
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef FKSTEERING_H
#define FKSTEERING_H

#include <QGpCoreMath.h>

#include "ArraySelection.h"
#include "FKPower.h"
#include "ArrayCoreDLLExport.h"

//#define FKSTEERING_1D

namespace ArrayCore {

  class FKPower;
  class FKParameters;

  class ARRAYCORE_EXPORT FKSteering
  {
  public:
    FKSteering(const ArraySelection * array);
    ~FKSteering();

    void setPhaseShifts(double step, double size);
    void setRadialRotations(double step);
    void setTransverseRotations(double step);
    void setSensorPositionErrors(const FKParameters& param);
    void setSensorOrientationErrors(const FKParameters& param);

    const ArraySelection * array() const {return _array;}

    const Vector<Complex>& coordinates() const {return _coordinates;}
    const Vector<double>& xCoordinates() const {return _xCoordinates;}
    const Vector<double>& yCoordinates() const {return _yCoordinates;}

    inline const Complex& rotation(const Vector<int>& index) const;
    inline const Complex * sensorOrientationErrors() const {return _sensorOrientationErrors;}

    inline const Vector<Complex>& oneComponent(const Vector<int>& index, FKPower * p) const;
    inline const Vector<Complex>& oneComponent(const Vector<double>& k, FKPower * p) const;

    inline const Vector<Complex>& twoComponentHorizontal(const Vector<int>& index, FKPower * p) const;
    inline const Vector<Complex>& twoComponentRadial(const Vector<double>& k, FKPower * p) const;
    inline const Vector<Complex>& twoComponentTransverse(const Vector<double>& k, FKPower * p) const;

    inline const Vector<Complex>& threeComponentRayleigh(const Vector<int>& index, FKPower * p) const;
    inline const Vector<Complex>& threeComponentRayleigh(const Vector<double>& k, FKPower * p) const;
    inline const Vector<Complex>& threeComponentLove(const Vector<int>& index, FKPower * p) const;
    inline const Vector<Complex>& threeComponentLove(const Vector<double>& k, FKPower * p) const;

    static inline const Vector<Complex>& twoComponentRayleighCross(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& twoComponentRayleighRadial(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& twoComponentRayleighVertical(const Complex& ell, FKPower * p);

    static inline const Vector<Complex>& threeComponentRayleighCross(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& threeComponentRayleighRadial(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& threeComponentRayleighVertical(const Complex& ell, FKPower * p);

    static inline const Vector<Complex>& twoComponentRayleighCrossGradient(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& twoComponentRayleighRadialGradient(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& twoComponentRayleighVerticalGradient(const Complex& ell, FKPower * p);

    static inline const Vector<Complex>& threeComponentRayleighCrossGradient(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& threeComponentRayleighRadialGradient(const Complex& ell, FKPower * p);
    static inline const Vector<Complex>& threeComponentRayleighVerticalGradient(const Complex& ell, FKPower * p);

    inline void oneComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const;
    inline void twoComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const;
    inline void threeComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const;
  private:
    const ArraySelection * _array;
    PrivateVector<Complex> _coordinates;
    PrivateVector<double> _xCoordinates;
    PrivateVector<double> _yCoordinates;

    int _kCount;
#ifdef FKSTEERING_1D
    Complex * _phaseShiftsX, * _phaseShiftsY;
#else
    Complex * _phaseShifts;
#endif
    Complex * _rotations;

    int _ellCount;
    Complex * _sensorOrientationErrors;
    bool _sensorPositionErrors;
  };

  inline const Complex& FKSteering::rotation(const Vector<int>& index) const
  {
    return *(_rotations+(index[1]*_kCount+index[0]));
  }

#ifdef FKSTEERING_1D
  inline const Vector<Complex>& FKSteering::oneComponent(const Vector<int>& index,
                                                         FKPower * p) const
  {
    Vector<Complex>& e=p->steeringVector();
    int n=_array->count();
    Complex * kx=_phaseShiftsX+index[0]*n;
    Complex * ky=_phaseShiftsY+index[1]*n;
    for(int i=0; i<n; i++) {
      Complex& v=e[i];
      v=kx[i];
      v*=ky[i];
    }
    p->setOneComponentSteeringVector(&e);
    return e;
  }
#else
  inline const Vector<Complex>& FKSteering::oneComponent(const Vector<int>& index,
                                                         FKPower * p) const
  {
    Vector<Complex>& e=p->steeringVectorGrid();
    e.setValues(_phaseShifts+(index[1]*_kCount+index[0])*_array->count());
    p->setOneComponentSteeringVector(&e);
    return e;
  }
#endif

  inline const Vector<Complex>& FKSteering::oneComponent(const Vector<double>& k,
                                                         FKPower * p) const
  {
    int n=_array->count();
    Vector<Complex>& e=p->steeringVector();
#if defined(__SSE3__) && !defined(Q_OS_DARWIN)
    // See .cpp for explanations
    Complex ck(-k[0], k[1]), x;
    for(int i=0; i<n; i++) {
      x=ck;
      x*=_coordinates[i];
      e[i].setUnitExp(x.re());
    }
#else
    for(int i=0; i<n; i++) {
      const Point2D& r=_array->relativePos(i);
      e[i].setUnitExp(-k.scalarProduct(r));
    }
#endif
    p->setOneComponentSteeringVector(&e);
    return e;
  }

  inline const Vector<Complex>& FKSteering::twoComponentHorizontal(const Vector<int>& index,
                                                                   FKPower * p) const
  {
    int n=_array->count();
    const Vector<Complex>& q=oneComponent(index, p);
    const Complex& rot=rotation(index);
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      ei=q[i];
      ein=ei;
      ei*=rot.re();
      ein*=rot.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRadial(const Vector<double>& k,
                                                               FKPower * p) const
  {
    int n=_array->count();
    Complex rotation(k[0], k[1]);
    rotation/=rotation.abs();
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      const Point2D& r=_array->relativePos(i);
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      ei.setUnitExp(-k.scalarProduct(r));
      ein=ei;
      ei*=rotation.re();
      ein*=rotation.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::twoComponentTransverse(const Vector<double>& k,
                                                                   FKPower * p) const
  {
    int n=_array->count();
    Complex rotation(-k[1], k[0]);
    rotation/=rotation.abs();
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      const Point2D& r=_array->relativePos(i);
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      ei.setUnitExp(-k.scalarProduct(r));
      ein=ei;
      ei*=rotation.re();
      ein*=rotation.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleigh(const Vector<int>& index,
                                                                   FKPower * p) const
  {
    // The elements are not multiplied by sin(xi) and cos(xi).
    int n=_array->count();
    int n2=n << 1;
    const Vector<Complex>& q=oneComponent(index, p);
    const Complex& rot=rotation(index);
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      Complex& ei2n=e[i+n2];
      ei2n=q[i];
      ei=ei2n;
      ein=ei;
      ei*=rot.re();
      ein*=rot.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleigh(const Vector<double>& k,
                                                                   FKPower * p) const
  {
    // The elements are not multiplied by sin(xi) and cos(xi).
    int n=_array->count();
    int n2=n << 1;
    Complex rotation(k[0], k[1]);
    double absRot=rotation.abs();
    if(absRot>0.0) {
      rotation/=absRot;
    } else {
      rotation=Complex(1.0, 0.0);
    }
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      const Point2D& r=_array->relativePos(i);
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      Complex& ei2n=e[i+n2];
      ei2n.setUnitExp(-k.scalarProduct(r));
      ei=ei2n;
      ein=ei;
      ei*=rotation.re();
      ein*=rotation.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::threeComponentLove(const Vector<int>& index,
                                                               FKPower * p) const
  {
    int n=_array->count();
    int n2=n << 1;
    const Vector<Complex>& q=oneComponent(index, p);
    const Complex& rot=rotation(index);
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      Complex& ei2n=e[i+n2];
      ei2n=Complex();
      ei=q[i];
      ein=ei;
      ei*=rot.re();
      ein*=rot.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::threeComponentLove(const Vector<double>& k,
                                                               FKPower * p) const
  {
    // The elements are not multiplied by sin(xi) and cos(xi). xi=90 deg for Love.
    int n=_array->count();
    int n2=n << 1;
    Complex rotation(-k[1], k[0]);
    double absRot=rotation.abs();
    if(absRot>0.0) {
      rotation/=absRot;
    } else {
      rotation=Complex(1.0, 0.0);
    }
    Vector<Complex>& e=p->steeringVector();
    for(int i=0; i<n; i++) {
      const Point2D& r=_array->relativePos(i);
      Complex& ei=e[i];
      Complex& ein=e[i+n];
      Complex& ei2n=e[i+n2];
      ei2n=Complex();
      ei.setUnitExp(-k.scalarProduct(r));
      ein=ei;
      ei*=rotation.re();
      ein*=rotation.im();
    }
    return e;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighCross(const Complex& ell,
                                                                      FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    double f=-ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      ei=q[i];
      ei.imaginaryMultiply(f);
      ein=q[i];
      ein*=ell.re();
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighRadial(const Complex& ell,
                                                                       FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    double f=-ell.im()/ell.re();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      ei=q[i];
      ein=ei;
      ei.imaginaryMultiply(f);
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighVertical(const Complex& ell,
                                                                         FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    double f=ell.re()/ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      ei=q[i];
      ein=ei;
      ei.imaginaryMultiply(-1.0);
      ein*=f;
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighCrossGradient(const Complex& ell,
                                                                              FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    double fh=-ell.re();
    double fz=-ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=edxi[i];
      Complex& ein=edxi[i+n];
      ei=q[i];
      ei.imaginaryMultiply(fh);
      ein=q[i];
      ein*=fz;
    }
    return edxi;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighRadialGradient(const Complex& ell,
                                                                               FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    double f=-1.0/(ell.re()*ell.re());
    for(int i=0; i<n; i++) {
      Complex& ei=edxi[i];
      edxi[i+n]=Complex();
      ei=q[i];
      ei.imaginaryMultiply(f);
    }
    return edxi;
  }

  inline const Vector<Complex>& FKSteering::twoComponentRayleighVerticalGradient(const Complex& ell,
                                                                                 FKPower * p)
  {
    const Vector<Complex>& q=p->oneComponentSteeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    double f=-1.0/(ell.im()*ell.im());
    for(int i=0; i<n; i++) {
      Complex& ein=edxi[i+n];
      edxi[i]=Complex();
      ein=q[i];
      ein*=f;
    }
    return edxi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighCross(const Complex& ell,
                                                                        FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double f=-ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      Complex& ei2n=exi[i+n2];
      ei=e[i];
      ei.imaginaryMultiply(f);
      ein=e[i+n];
      ein.imaginaryMultiply(f);
      ei2n=e[i+n2];
      ei2n*=ell.re();
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighRadial(const Complex& ell,
                                                                         FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double f=-ell.im()/ell.re();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      Complex& ei2n=exi[i+n2];
      ei=e[i];
      ei.imaginaryMultiply(f);
      ein=e[i+n];
      ein.imaginaryMultiply(f);
      ei2n=e[i+n2];
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighVertical(const Complex& ell,
                                                                           FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& exi=p->ellipticitySteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double f=ell.re()/ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=exi[i];
      Complex& ein=exi[i+n];
      Complex& ei2n=exi[i+n2];
      ei=e[i];
      ei.imaginaryMultiply(-1.0);
      ein=e[i+n];
      ein.imaginaryMultiply(-1.0);
      ei2n=e[i+n2];
      ei2n*=f;
    }
    return exi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighCrossGradient(const Complex& ell,
                                                                                FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double fh=-ell.re();
    double fz=-ell.im();
    for(int i=0; i<n; i++) {
      Complex& ei=edxi[i];
      Complex& ein=edxi[i+n];
      Complex& ei2n=edxi[i+n2];
      ei=e[i];
      ei.imaginaryMultiply(fh);
      ein=e[i+n];
      ein.imaginaryMultiply(fh);
      ei2n=e[i+n2];
      ei2n*=fz;
    }
    return edxi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighRadialGradient(const Complex& ell,
                                                                                 FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double f=-1.0/(ell.re()*ell.re());
    for(int i=0; i<n; i++) {
      Complex& ei=edxi[i];
      Complex& ein=edxi[i+n];
      ei=e[i];
      ei.imaginaryMultiply(f);
      ein=e[i+n];
      ein.imaginaryMultiply(f);
      edxi[i+n2]=Complex();
    }
    return edxi;
  }

  inline const Vector<Complex>& FKSteering::threeComponentRayleighVerticalGradient(const Complex& ell,
                                                                                 FKPower * p)
  {
    const Vector<Complex>& e=p->steeringVector();
    Vector<Complex>& edxi=p->ellipticityDerivativeSteeringVector();
    int n=p->stationCount();
    int n2=n << 1;
    double f=-1.0/(ell.im()*ell.im());
    for(int i=0; i<n; i++) {
      Complex& ei2n=edxi[i+n2];
      edxi[i]=Complex();
      edxi[i+n]=Complex();
      ei2n=e[i+n2];
      ei2n*=f;
    }
    return edxi;
  }

  inline void FKSteering::oneComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const
  {
    Vector<double>& r=p->sensorProjections();
    Complex k1(k[0], -k[1]);
    k1/=k1.abs();
    int n=_array->count();
    for(int i=0; i<n; i++) {
      double& ri=r[i];
      ri=(k1*_coordinates[i]).re();
    }
  }

  inline void FKSteering::twoComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const
  {
    Vector<double>& r=p->sensorProjections();
    Complex k1(k[0], -k[1]);
    k1/=k1.abs();
    int n=_array->count();
    for(int i=0; i<n; i++) {
      double& ri=r[i];
      ri=(k1*_coordinates[i]).re();
      r[i+n]=ri;
    }
  }

  inline void FKSteering::threeComponentSensorRadialProjections(const Vector<double>& k, FKPower * p) const
  {
    Vector<double>& r=p->sensorProjections();
    Complex k1(k[0], -k[1]);
    k1/=k1.abs();
    int n=_array->count();
    int n2=2*n;
    for(int i=0; i<n; i++) {
      double& ri=r[i];
      ri=(k1*_coordinates[i]).re();
      r[i+n]=ri;
      r[i+n2]=ri;
    }
  }

} // namespace ArrayCore

#endif // FKSTEERING_H

