/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-07-21
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreMath.h>

#include "ConvThreeComponentXiFunction.h"

namespace ArrayCore {

  /*!
    \class ConvThreeComponentXiFunction ConvThreeComponentXiFunction.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ConvThreeComponentXiFunction::ConvThreeComponentXiFunction()
  {
    _tx=0.0;
    _ellh=0.0;
    _ellz=0.0;
    _invellz=0.0;
    _xix=0.0;
    _cxix=0.0;
    _sxix=0.0;
    _cxix2=0.0;
    _sxix2=0.0;
    _precision=M_PI/18000.0;
    _xir1=std::numeric_limits<double>::quiet_NaN();
    _xir2=std::numeric_limits<double>::quiet_NaN();
  }

  void ConvThreeComponentXiFunction::setParameters(double tx, double ellh, double ellz, double xix)
  {
    QFile f("/tmp/data");
    f.open(QIODevice::Append);
    QTextStream s(&f);
    s << "tx " << tx
      << " xi " << Angle::radiansToDegrees(xix)
      << " xih " << Angle::radiansToDegrees(atan(ellh))
      << " xiz " << Angle::radiansToDegrees(atan(ellz))
      << Qt::endl;
    f.close();
    _tx=tx;
    _ellh=ellh;
    _ellz=ellz;
    _invellz=1.0/_ellz;
    _xix=xix;
    _cxix=cos(_xix);
    _sxix=sin(_xix);
    _cxix2=_cxix*_cxix;
    _sxix2=_sxix*_sxix;
  }

  double ConvThreeComponentXiFunction::value(double xi) const
  {
    double cxi=cos(xi);
    double sxi=sin(xi);
    double cxi2=cxi*cxi;
    double sxi2=sxi*sxi;
    double csxi=cxi*sxi;
    double RhN=_ellh*csxi-sxi2;
    double RzN=_invellz*csxi-cxi2;
    double d=_cxix*cxi+_sxix*sxi;
    double tx=1.0+(RhN*_sxix2+RzN*_cxix2)/(d*d);
    return _tx-tx;
  }

  double ConvThreeComponentXiFunction::verticalNoise(double xir) const
  {
    double cxir=cos(xir);
    double sxir=sin(xir);
    return cxir*(sxir*_invellz-cxir);
  }

  double ConvThreeComponentXiFunction::horizontalNoise(double xir) const
  {
    double cxir=cos(xir);
    double sxir=sin(xir);
    return sxir*(cxir*_ellh-sxir);
  }

  void ConvThreeComponentXiFunction::solve()
  {
    printFunction();
    double xiz=atan(_ellz);
    double xih=atan(_ellh);
    double y1, y2;

    y1=value(xiz);
    y2=value(xih);
    if(y1*y2<0.0) {
      _xir1=refineRoot(xiz, y1, xih);
      _xir2=std::numeric_limits<double>::quiet_NaN();
      //check();
      return;
    }

    double dxi=0.01*(xih-xiz);
    double xi=xiz;
    double minxi=xi;
    double miny=y1;
    for(xi+=dxi; fabs(xi)<fabs(xih); xi+=dxi) {
      y2=value(xi);
      if(y2<0.0) {
        _xir1=refineRoot(xi-dxi, y1, xi);
        _xir2=refineRoot(xi, y2, xih);
        check();
        return;
      }
      if(y2<miny) {
        minxi=xi;
        miny=y2;
      }
      y1=y2;
    }
    refineMinimum(minxi-dxi, minxi+dxi);
    if(value(_xir1)<0.0) {
      APP_LOG(1, tr("Negative value at single solution, refining roots\n"))
      _xir1=refineRoot(minxi-dxi, value(minxi-dxi), _xir1);
      _xir2=refineRoot(_xir1, value(_xir1), minxi+dxi);
    }
    check();
  }

  void ConvThreeComponentXiFunction::check()
  {
    double xi;
    double xiz=atan(_ellz);
    double xih=atan(_ellh);
    double dxi;
    double err=false;
    if(xiz<0.0) {
      dxi=-_precision;
    } else {
      dxi=_precision;
    }
    if(isnormal(_xir1)) {
      for(xi=xiz+dxi; fabs(xi)<fabs(_xir1)-_precision; xi+=dxi) {
        if(value(xi)<0.0) {
          APP_LOG(1, tr("Negative value at %1 between xiz(%2) and xir1(%3)\n")
                  .arg(Angle::radiansToDegrees(xi))
                  .arg(Angle::radiansToDegrees(xiz))
                  .arg(Angle::radiansToDegrees(_xir1)));
          err=true;
          break;
        }
      }
    }
    if(isnormal(_xir2)) {
      for(xi=_xir1+dxi; fabs(xi)<fabs(_xir2)-_precision; xi+=dxi) {
        if(value(xi)>0.0) {
          APP_LOG(1, tr("Positive value at %1 between xir1(%2) and xir2(%3)\n")
                  .arg(Angle::radiansToDegrees(xi))
                  .arg(Angle::radiansToDegrees(_xir1))
                  .arg(Angle::radiansToDegrees(_xir2)));
          err=true;
          break;
        }
      }
      for(xi=_xir2+dxi; fabs(xi)<fabs(xih); xi+=dxi) {
        if(value(xi)<0.0) {
          APP_LOG(1, tr("Negative value at %1 between xir2(%2) and xih(%3)\n")
                  .arg(Angle::radiansToDegrees(xi))
                  .arg(Angle::radiansToDegrees(_xir2))
                  .arg(Angle::radiansToDegrees(xih)));
          err=true;
          break;
        }
      }
    } else {
      for(xi=_xir1+dxi; fabs(xi)<fabs(xih); xi+=dxi) {
        if(value(xi)<0.0) {
          APP_LOG(1, tr("Negative value at %1 between xir1(%2) and xih(%3)\n")
                  .arg(Angle::radiansToDegrees(xi))
                  .arg(Angle::radiansToDegrees(_xir1))
                  .arg(Angle::radiansToDegrees(xih)));
          err=true;
          break;
        }
      }
    }
    if(err || App::verbosity()>=1) {
      dxi=0.01*(xih-xiz);
      for(xi=xiz; fabs(xi)<fabs(xih); xi+=dxi) {
        printf("%lf %lf\n", Angle::radiansToDegrees(xi), value(xi));
      }
      printf("%lf %lf\n", Angle::radiansToDegrees(xih), value(xih));
      fflush(stdout);
    }
  }

  double ConvThreeComponentXiFunction::refineRoot(double xi1, double f1, double xi2) const
  {
    double xi3=0.5*(xi1+xi2), f3;
    while(fabs(xi1-xi2)>_precision) {
      f3=value(xi3);
      if(f1*f3<0.0) {
        xi2=xi3;
      } else {
        xi1=xi3;
        f1=f3;
      }
      xi3=0.5*(xi1+xi2);
    }
    return xi3;
  }

  void ConvThreeComponentXiFunction::refineMinimum(double xi1, double xi2)
  {
    double dxi=0.01*(xi2-xi1);

    double xi=xi1;
    double minxi=xi;
    double miny=value(minxi);
    for(xi+=dxi; fabs(xi)<fabs(xi2); xi+=dxi) {
      double y=value(xi);
      if(y<miny) {
        minxi=xi;
        miny=y;
      }
    }
    if(dxi<_precision) {
      _xir1=minxi;
      _xir2=std::numeric_limits<double>::quiet_NaN();
    } else {
      refineMinimum(minxi-dxi, minxi+dxi);
    }
  }

  void ConvThreeComponentXiFunction::printFunction() const
  {
    // The order of ellz and ellh absolute values is not
    // the same as for high resolution.
    double xiz=atan(_ellz);
    double xih=atan(_ellh);
    double dxi=0.01*(xih-xiz);

    QFile f("/tmp/curve");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);
    double xi=xiz;
    for(xi+=dxi; fabs(xi)<fabs(xih); xi+=dxi) {
      s << Angle::radiansToDegrees(xi) << " " << value(xi) << Qt::endl;
    }
    f.close();
  }

} // namespace ArrayCore

