/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-09-17
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveHRFKRayleigh.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class ActiveHRFKRayleigh ActiveHRFKRayleigh.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  bool ActiveHRFKRayleigh::invertCrossSpectrum()
  {
    return _crossSpectrum->invert();
  }

  inline void ActiveHRFKRayleigh::maximumEllipticity() const
  {
    // We admit values ouside -pi/2 to p/2 to avoid been trapped right at +/- pi/2
    // This happens if _xi is between the limits and the minimum
    _ellOptimization.maximize(_xi, this, M_PI/8.0);
  }

  double ActiveHRFKRayleigh::value(Vector<double>& kell) const
  {
    kell[1]=0.0; // Only k[0] is modified by GridSearch
    if(isInsideLimits(kell)) {
      _steering->oneComponent(kell[0], powerEngine());
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void ActiveHRFKRayleigh::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighCross(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.value=powerEngine()->value();
  }

  void ActiveHRFKRayleigh::setFunctionDerivatives(Attributes& a) const
  {
    double value2=a.value*a.value;
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighCrossGradient(a.ell, powerEngine());
    a.slope=-2.0*value2*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.slope*a.slope/a.value+a.value-value2*a.concavity);
  }

} // namespace ArrayCore

