/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef AbstractFKFunction_H
#define AbstractFKFunction_H

#include <QGpCoreMath.h>

#include "FKPower.h"
#include "FKCache.h"
#include "FKPeaks.h"
#include "ArrayStationSignals.h"
#include "ArrayCoreDLLExport.h"

#define FK_PASSIVE_STEERING \
  FKSteering * steeringInitialization() {return cache()->passiveSteering();} \
  const FKSteering * steering() const {return cache()->passiveSteering();} \
  inline const Vector<Complex>& oneComponentSteering(const Vector<int>& index) const { \
    return steering()->oneComponent(index, powerEngine()); \
  } \
  inline const Vector<Complex>& oneComponentSteering(const Vector<double>& k) const { \
    return steering()->oneComponent(k, powerEngine()); \
  }

#define FK_ACTIVE_STEERING \
    inline void oneComponentSteering(double k) const { \
      _steering->oneComponent(k, powerEngine()); \
    } \
  protected: \
    ActiveFKSteering * _steering;

namespace ArrayCore {

  class FKLoopWorker;
  class FKGradientSearch;
  class FKCrossSpectrum;
  class FKPeaks;
  class WaveNumberConverter;

  class ARRAYCORE_EXPORT AbstractFKFunction : public AbstractFunction
  {
  public:
    AbstractFKFunction(FKCache * cache);
    ~AbstractFKFunction();

    void setMaximumSlowness(double slow) {_squaredSlowMax=slow*slow;}
    double maximumSlowness() const {return sqrt(_squaredSlowMax);}

    void setMinimumSlowness(double slow) {_squaredSlowMin=slow*slow;}
    double minimumSlowness() const {return sqrt(_squaredSlowMin);}

    void setMinimumWavenumber(double k) {_squaredKminArray=k*k;}
    void setMaximumWavenumber(double k) {_squaredKmaxArray=k*k;}

    void setTotalLimits(double squaredOmega);
    void setAzimuthRange(double min, double max);

    virtual void initialize(const FKParameters& param)=0;
    virtual FunctionSearch * createSearch(const FKParameters& param);

    virtual bool setSource(const VectorList<int>& /*blocks*/) {return true;}

    virtual void resetCrossSpectrum()=0;
    virtual void addCrossSpectrum()=0;
    virtual void meanCrossSpectrum(int nBlocks)=0;
    virtual void addIncoherentNoise(double /*horizontal*/, double /*vertical*/) {}
    virtual bool invertCrossSpectrum() {return true;}
    virtual bool setFixedEllipticity(double /*frequency*/) {return true;}
    virtual double wavenumberNormalizedConcavity(const Vector<double>& /*k*/) const {return 0.0;}

    virtual void setAssociatedResults(const Vector<double>& kell, double power,
                                      FKPeaks::Value& val) const;

    virtual bool remove(const Vector<double>& /*k*/, FKPeaks::Value& /*val*/) {return false;}

    virtual bool isOnLimit(const Vector<double>& k, double frequency) const;
    inline bool isInsideLimits(const Vector<int>& index) const;
    inline bool isInsideLimits(const Vector<double>& k) const;

    virtual void setCrossSpectrum(FKCrossSpectrum * s) {_crossSpectrum=s;}
    void setCrossSpectrum(const ComplexMatrix * f) {_powerEngine->setCrossSpectrum(f);}
    virtual const ComplexMatrix * crossSpectrum() const;

    static AbstractFKFunction * create(ArrayStations::Mode mode,
                                       const FKParameters * param,
                                       FKCache * cache);
    void setGridStep(double step) {_gridStep=step;}
  protected:
    FKCache * cache() const {return _cache;}

    void setPowerEngine(FKPower * p);
    FKPower * powerEngine() const {return _powerEngine;}

    FKCrossSpectrum * _crossSpectrum;
  private:
    FKCache * _cache;
    FKPower * _powerEngine;
    double _gridStep;
    double _squaredKminArray;
    double _squaredKmaxArray;
    double _squaredSlowMin;
    double _squaredSlowMax;
    double _squaredKminTotal, _squaredKmaxTotal;
    double _gridSquaredKminTotal, _gridSquaredKmaxTotal;
    double _minAzimuth, _maxAzimuth;
  };

  inline bool AbstractFKFunction::isInsideLimits(const Vector<int>& index) const
  {
    double k2=index.squareLength(0, 1);
    if(k2<_gridSquaredKminTotal || k2>_gridSquaredKmaxTotal) {
      return false;
    }
    double a=atan2(index[1], index[0]); // from -pi/2 to pi/2
    return Angle::isInsideRadians(a, _minAzimuth, _maxAzimuth);
  }

  inline bool AbstractFKFunction::isInsideLimits(const Vector<double>& k) const
  {
    double k2=k.squareLength(0, 1);
    if(k2<_squaredKminTotal || k2>_squaredKmaxTotal) {
      return false;
    }
    double a=atan2(k[1], k[0]); // from -pi/2 to pi/2
    return Angle::isInsideRadians(a, _minAzimuth, _maxAzimuth);
  }

  template <class Function, class Attributes> void printEllipticityCurve(const Function * func)
  {
    QFile f("/tmp/ellderivatives");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);
    Attributes a;
    for(a.x=-M_PI/2; a.x<=M_PI/2; a.x+=M_PI/100) {
      func->setFunctionValue(a);
      func->setFunctionDerivatives(a);
      s << a.x << " " << a.value << " " << a.slope << " " << a.concavity << Qt::endl;
    }
    f.close();
  }

} // namespace ArrayCore

#endif // AbstractFKFunction_H
