/***************************************************************************
**
**  This file is part of waran.
**
**  waran is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  waran is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2012-02-08
**  Copyright: 2012-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "GpsLink.h"
#include "Station.h"

/*!
  \class GpsLink GpsLink.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
GpsLink::GpsLink(Station * station)
  : DaemonLink(station)
{
  TRACE;
  setPort(2974);
  _refresh.setInterval(2000);
  QObject::connect(&_refresh, SIGNAL(timeout()), this, SLOT(refreshVariableParameters()));
  _refresh.start();
  _rawMode=false;
  _rawRecording=0;
}

GpsLink::~GpsLink()
{
  TRACE;
  delete _rawRecording;
}

void GpsLink::connected()
{
  TRACE;
  DaemonLink::connected();
  refreshVariableParameters();
}

void GpsLink::refreshVariableParameters()
{
  if(_rawRecording) {
    station()->setGpsState(QString("recording %1").arg(Number::secondsToDuration(_rawRecording->elapsed()/1000, 0)));
  } else if(isAvailable()) {
    state();
  }
}

void GpsLink::highRateRefresh()
{
  _refresh.setInterval(2000); // 2 s during startup
}

/*!

*/
int GpsLink::bytesAvailable(const char * buffer, int bytesCount)
{
  TRACE;
  if(_rawMode) {
    return saveRaw(buffer, bytesCount);
  }
  int bytesRead=0;
  while(bytesRead<bytesCount) {
    // Scan for a line
    int newBytesRead;
    for(newBytesRead=bytesRead; newBytesRead<bytesCount && buffer[newBytesRead]!='\n'; newBytesRead++) {}
    if(newBytesRead==bytesCount) {
      return bytesRead; // partial line
    }
    switch(buffer[bytesRead]) {
    case 'r':
      if(match(buffer, bytesRead, bytesCount, "raw begin")) {
        _rawMode=true;
        newBytesRead++;
        return newBytesRead+saveRaw(buffer+newBytesRead, bytesCount-newBytesRead);
      }
      break;
    case 's':
      if(match(buffer, bytesRead, bytesCount, "state=")) {
        state(buffer, bytesRead, newBytesRead);
      }
      break;
    default:
      break;
    }
    bytesRead=newBytesRead+1;
    // Skip blanks and additional end of line characters
    while(bytesRead<bytesCount && isspace(buffer[bytesRead])) {bytesRead++;}
  }
  return bytesRead;
}

void GpsLink::state(const char * buffer, int bytesRead, int bytesCount)
{
  TRACE;
  if(match(buffer, bytesRead, bytesCount, "recording ")) {
    bool ok;
    long int value=readLongInteger(buffer, bytesRead, bytesCount, ok);
    if(value>0 && _refresh.interval()<10000) { // Reduce refresh rate (once every 10 s)
      _refresh.setInterval(10000);
    }
    if(ok) {
      station()->setGpsState(QString("recording %1").arg(Number::secondsToDuration(value, 0)));
    }
  } else {
    delete _rawRecording;
    _rawRecording=0;
    station()->setGpsState(QByteArray(buffer+bytesRead, bytesCount-bytesRead));
  }
}

int GpsLink::saveRaw(const char * buffer, int bytesCount)
{
  TRACE;
  int bytesRead=0;
  switch(findMatch(buffer, bytesRead, bytesCount, "raw checksum=")) {
  default: // checksum not found at all
    _rawFileSize+=bytesRead;
    _rawChecksum.add(buffer, bytesRead);
    _rawFile.write(buffer, bytesRead);
    emit rawFileSize(_rawFileSize);
    return bytesRead;
  case 1: // wait for more bytes, checksum found at the end but truncated
    return bytesRead;
  case 2: { // checksum found completely in buffer
      bool ok;
      int rawBytes=bytesRead-13;
      QByteArray value=readString(buffer, bytesRead, bytesCount, ok);
      if(ok) {
        value=QByteArray::fromHex(value);
        unsigned char a=value.data()[0];
        unsigned char b=value.data()[1];
        _rawFileSize+=rawBytes;
        _rawChecksum.add(buffer, rawBytes);
        _rawFile.write(buffer, rawBytes);
        _rawFile.close();
        emit rawFileSize(_rawFileSize);
        if(FletcherChecksum(a, b)==_rawChecksum) {
          emit rawFileReady();
        } else {
          emit rawFileError();
        }
        _rawMode=false;
        return bytesRead;
      } else {
        // wait for more bytes, checksum found at the end but truncated
        return bytesRead;
      }
    }
  }
}

void GpsLink::start(int duration)
{
  TRACE;
  QByteArray buf("start=");
  buf+=QByteArray::number(duration);
  buf+="\n";
  send(buf.data());
  if(duration>0) {
    delete _rawRecording;
    _rawRecording=new QElapsedTimer;
    _rawRecording->start();
  }
}

void GpsLink::stop()
{
  TRACE;
  send("stop\n");
}

void GpsLink::navigation()
{
  TRACE;
  send("navigation\n");
}

void GpsLink::state()
{
  TRACE;
  send("state\n");
}

void GpsLink::raw()
{
  TRACE;
  QString fileName=station()->gpsFileName();
  fileName+="ubx";
  _rawFile.setFileName(fileName);
  _rawFileSize=0;
  _rawChecksum.reset();
  if(_rawFile.open(QIODevice::WriteOnly)) {
    send("raw\n");
  } else {
    App::log(tr("Cannot open file %1 for writing\n").arg(fileName) );
  }
}

void GpsLink::abortRaw()
{
  TRACE;
  _rawFile.close();
  _rawFile.setFileName(QString());
}

