/***************************************************************************
**
**  This file is part of phaseit.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-08-09
**  Authors:
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Parameters.h"

/*!
  \class Parameters Parameters.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

Parameters::Parameters()
{
  TRACE;
  _frequencyBandwidth=0.1;
  _maximumDelay=0.5;
  _referenceIndex=0;
  _domain=Frequency;

  _windowing.setLength(50.0);
  _frequencySampling.setRange(0.2, 20.0);
  _frequencySampling.setScaleType(SamplingParameters::Log);
  _frequencySampling.setStep(1.025);
}

Parameters::Parameters(const Parameters& o)
  : AbstractParameters(o),
    _timeRange(o._timeRange),
    _windowing(o._windowing),
    _frequencySampling(o._frequencySampling)
{
  TRACE;
  _frequencyBandwidth=o._frequencyBandwidth;
  _maximumDelay=o._maximumDelay;
  _referenceIndex=o._referenceIndex;
  _domain=o._domain;
}

void Parameters::setDomain(const QString& d)
{
  TRACE;
  QString dl=d.toLower();
  if(dl=="frequency") {
    _domain=Frequency;
  } else if(dl=="time"){
    _domain=Time;
  }
}

QString Parameters::domainString() const
{
  TRACE;
  switch(_domain) {
  case Frequency:
    break;
  case Time:
    return "Time";
  }
  return "Frequency";
}

int Parameters::keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const
{
  return 4+AbstractParameters::keywordCount();
}

int Parameters::totalKeywordCount(PARAMETERS_TOTALKEYWORDCOUNT_ARGS) const
{
  return AbstractParameters::totalKeywordCount()+
      _timeRange.totalKeywordCount()+
      _windowing.totalKeywordCount()+
      _frequencySampling.totalKeywordCount();
}

void Parameters::collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS)
{
  TRACE;
  int baseIndex=AbstractParameters::keywordCount();
  keywords.add(prefix+"FREQ_BAND_WIDTH"+suffix, this, baseIndex+1);
  keywords.add(prefix+"MAXIMUM_DELAY"+suffix, this, baseIndex+2);
  keywords.add(prefix+"REFERENCE_INDEX"+suffix, this, baseIndex+3);
  keywords.add(prefix+"DOMAIN"+suffix, this, baseIndex+4);
  _timeRange.collectKeywords(keywords, prefix, suffix);
  _windowing.collectKeywords(keywords, prefix, suffix);
  _frequencySampling.collectKeywords(keywords, prefix, "_FREQUENCY"+suffix);
}

QString Parameters::toString(PARAMETERS_TOSTRING_ARGS_IMPL) const
{
  TRACE;
  QString log;
  log+=_timeRange.toString(prefix, suffix);
  log+=_windowing.toString(prefix, suffix);
  log+=_frequencySampling.toString(prefix, "_FREQUENCY"+suffix);
  log+=prefix+"FREQ_BAND_WIDTH"+suffix+"="+QString::number(_frequencyBandwidth)+"\n";
  log+="# Index of the reference station\n";
  log+=prefix+"REFERENCE_INDEX"+suffix+"="+QString::number(_referenceIndex)+"\n";
  log+="# Possible values for DOMAIN: Frequency or Time\n";
  log+=prefix+"DOMAIN"+suffix+"="+domainString()+"\n";
  log+="# Correlation delay for time domain processing\n";
  log+=prefix+"MAXIMUM_DELAY"+suffix+"="+QString::number(_maximumDelay)+"\n";
  return log;
}

bool Parameters::setValue(PARAMETERS_SETVALUE_ARGS)
{
  TRACE;
  bool ok=true;
  switch(index-AbstractParameters::keywordCount()) {
  case 0:
    _frequencyBandwidth=value.toDouble(&ok);
    return ok;
  case 1:
    _maximumDelay=value.toDouble(&ok);
    return ok;
  case 2:
    _referenceIndex=value.toInt(&ok);
    if(_referenceIndex<0) {
      _referenceIndex=0;
      ok=false;
    }
    return ok;
  case 3:
    setDomain(value);
    return true;
  default:
    break;
  }
  return AbstractParameters::setValue(index, value, unit, keywords);
}

bool Parameters::isValid(int stationCount) const
{
  TRACE;
  if(_referenceIndex>=stationCount) {
    App::log(tr("Reference index (%1) is not compatible with the number of stations (%1)\n")
             .arg(_referenceIndex).arg(stationCount));
    return false;
  } else {
    return true;
  }
}
