/***************************************************************************
**
**  This file is part of gpviewmax.
**
**  gpviewmax is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpviewmax is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-07-19
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Samples.h"

/*!
  \class Samples Samples.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

Samples::Samples(const Samples& o)
  : VectorList<Sample>()
{
  _headerLine=o._headerLine;
}

bool Samples::lessThan(const Sample& s1, const Sample& s2)
{
  if(s1.frequency()<s2.frequency()) {
    return true;
  } else if(s1.frequency()>s2.frequency()) {
    return false;
  } else {
    if(s1.time()<s2.time()) {
      return true;
    } else {
      return false;
    }
  }
}

bool Samples::lessThanSlowness(const Sample& s1, const Sample& s2)
{
  if(s1.frequency()<s2.frequency()) {
    return true;
  } else if(s1.frequency()>s2.frequency()) {
    return false;
  } else {
    if(s1.slowness()<s2.slowness()) {
      return true;
    } else {
      return false;
    }
  }
}

Samples * Samples::filterUndefinedEllipticities() const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out undefined ellipticities"));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(isfinite(s.ellipticity()) && s.ellipticity()!=0.0) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumEllipticities(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out ellipticity below %1 deg.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.ellipticity()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumEllipticities(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out ellipticity above %1 deg.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.ellipticity()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumVerticalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out vertical noise below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.verticalNoise()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumVerticalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out vertical noise above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.verticalNoise()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumHorizontalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out horizontal noise below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.horizontalNoise()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumHorizontalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out horizontal noise above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.horizontalNoise()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumTotalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out total noise below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.totalNoise()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumTotalNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out total noise above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.totalNoise()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumDeltaNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out delta noise below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.deltaNoise()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumDeltaNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out delta noise above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.deltaNoise()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumSigmaNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out sigma noise below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.sigmaNoise()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumSigmaNoise(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out sigma noise above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.sigmaNoise()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterAbnormalSigmaNoise() const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out abnormal sigma noise (<0.01 or >100)."));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.sigmaNoise()>0.01 && s.sigmaNoise()<100.0) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumAzimuth(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out azimuth below %1 deg.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.azimuth()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumAzimuth(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out azimuth above %1 deg.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.azimuth()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}
Samples * Samples::filterMinimumPower(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out power below %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.power()>threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMaximumPower(double threshold) const
{
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out power above %1.").arg(threshold));
  Samples * newSamples=new Samples(*this);
  int n=count();
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(s.power()<threshold) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

Samples * Samples::filterMinimumVelocity(double threshold) const
{
  if(threshold>0.0) {
    ConsoleProgress progress;
    progress.setCaption(tr("Filtering out velocity below %1 m/s").arg(threshold));
    Samples * newSamples=new Samples(*this);
    threshold=1.0/threshold;
    int n=count();
    for(int i=0; i<n; i++) {
      const Sample& s=at(i);
      if(s.slowness()<threshold) {
        newSamples->append(s);
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

Samples * Samples::filterMaximumVelocity(double threshold) const
{
  if(threshold<std::numeric_limits<double>::infinity()) {
    ConsoleProgress progress;
    progress.setCaption(tr("Filtering out velocity above %1 m/s").arg(threshold));
    Samples * newSamples=new Samples(*this);
    threshold=1.0/threshold;
    int n=count();
    for(int i=0; i<n; i++) {
      const Sample& s=at(i);
      if(s.slowness()>threshold) {
        newSamples->append(s);
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

Samples * Samples::filterMinimumWavenumber(double threshold) const
{
  if(threshold>0.0) {
    ConsoleProgress progress;
    progress.setCaption(tr("Filtering out wavenumber below %1 rad/m").arg(threshold));
    Samples * newSamples=new Samples(*this);
    int n=count();
    for(int i=0; i<n; i++) {
      const Sample& s=at(i);
      if(2.0*M_PI*s.frequency()*s.slowness()>threshold) {
        newSamples->append(s);
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

Samples * Samples::filterMaximumWavenumber(double threshold) const
{
  if(threshold<std::numeric_limits<double>::infinity()) {
    ConsoleProgress progress;
    progress.setCaption(tr("Filtering out wavenumber above %1 rad/m").arg(threshold));
    Samples * newSamples=new Samples(*this);
    int n=count();
    for(int i=0; i<n; i++) {
      const Sample& s=at(i);
      if(2.0*M_PI*s.frequency()*s.slowness()<threshold) {
        newSamples->append(s);
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

Samples * Samples::filterNoiseSmart(double noiseDev, double slowDev) const
{
  if(noiseDev<std::numeric_limits<double>::infinity()) {
    ConsoleProgress progress;
    progress.setCaption(tr("Smart filtering noise, slowness deviation %1, noise deviation %2")
                        .arg(slowDev).arg(noiseDev));
    Samples sortedSamples(*this); // Work on a copy to keep constness
    std::sort(sortedSamples.begin(), sortedSamples.end(), lessThanSlowness);
    Samples * newSamples=new Samples(*this);
    int n=sortedSamples.count(), j;
    double thSlow, thNoise, freq;
    bool acceptSample=true;
    for(int i=0; i<n; i++) {
      const Sample& s=sortedSamples.at(i);
      freq=s.frequency();
      thNoise=s.verticalNoise()/(1.0+noiseDev);
      thSlow=s.slowness()/(1.0+slowDev);
      j=i-1;
      while(j>=0) {
        const Sample& st=sortedSamples.at(j);
        if(st.frequency()!=freq) {
          break;
        }
        if(st.slowness()<thSlow) {
          break;
        }
        if(st.verticalNoise()<thNoise) {
          acceptSample=false;
          break;
        }
        j--;
      }
      if(acceptSample) {
        thSlow=s.slowness()*(1.0+slowDev);
        j=i+1;
        while(j<n) {
          const Sample& st=sortedSamples.at(j);
          if(st.frequency()!=freq) {
            break;
          }
          if(st.slowness()>thSlow) {
            break;
          }
          if(st.verticalNoise()<thNoise) {
            acceptSample=false;
            break;
          }
          j++;
        }
        if(acceptSample) {
          newSamples->append(s);
        } else {
          acceptSample=true;
        }
      } else {
        acceptSample=true;
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}
Samples * Samples::filterPowerSmart(double powerDev, double slowDev) const
{
  if(powerDev<std::numeric_limits<double>::infinity()) {
    ConsoleProgress progress;
    progress.setCaption(tr("Smart filtering power, slowness deviation %1, power deviation %2")
                        .arg(slowDev).arg(powerDev));
    Samples sortedSamples(*this); // Work on a copy to keep constness
    std::sort(sortedSamples.begin(), sortedSamples.end(), lessThanSlowness);
    Samples * newSamples=new Samples(*this);
    int n=sortedSamples.count(), j;
    double thSlow, thPower, freq;
    bool acceptSample=true;
    for(int i=0; i<n; i++) {
      const Sample& s=sortedSamples.at(i);
      freq=s.frequency();
      thPower=s.power()*(1.0+powerDev);
      // First check lower slownesses
      thSlow=s.slowness()/(1.0+slowDev);
      j=i-1;
      while(j>=0) {
        const Sample& st=sortedSamples.at(j);
        if(st.frequency()!=freq) {
          break;
        }
        if(st.slowness()<thSlow) {
          break;
        }
        if(st.power()>thPower) { // In the slowness neighborhood, if there is at least
                                 // one sample with a sufficiently high power
                                 // ---> reject the current sample
          acceptSample=false;
          break;
        }
        j--;
      }
      if(acceptSample) {
        // Secondly, check higher slownesses
        thSlow=s.slowness()*(1.0+slowDev);
        j=i+1;
        while(j<n) {
          const Sample& st=sortedSamples.at(j);
          if(st.frequency()!=freq) {
            break;
          }
          if(st.slowness()>thSlow) {
            break;
          }
          if(st.power()>thPower) { // In the slowness neighborhood, if there is at least
                                   // one sample with a sufficiently high power
                                   // ---> reject the current sample
            acceptSample=false;
            break;
          }
          j++;
        }
        if(acceptSample) {
          newSamples->append(s);
        } else {
          acceptSample=true;
        }
      } else {
        acceptSample=true;
      }
      progress.setValue(i);
    }
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

Samples * Samples::filterRelativePower(double threshold) const
{
  if(threshold>0.0 &&
     !isEmpty()) {  // first() used below
    ConsoleProgress progress;
    progress.setCaption(tr("Filtering out relative power below %1").arg(threshold));
    Samples sortedSamples(*this); // Work on a copy to keep constness
    std::sort(sortedSamples.begin(), sortedSamples.end(), lessThan);
    Samples * newSamples=new Samples(*this);
    QString time0=sortedSamples.first().time();
    double frequency0=sortedSamples.first().frequency();
    double maxPower=0.0;
    int firstSampleIndex=0;
    int n=sortedSamples.count();
    for(int i=0; i<n; i++) {
      const Sample& s=sortedSamples.at(i);
      if(s.time()==time0 && s.frequency()==frequency0) {
        if(s.power()>maxPower) {
          maxPower=s.power();
        }
      } else {
        sortedSamples.selectPower(*newSamples, firstSampleIndex, i, maxPower*threshold);
        maxPower=s.power();
        firstSampleIndex=i;
        time0=s.time();
        frequency0=s.frequency();
      }
      progress.setValue(i);
    }
    sortedSamples.selectPower(*newSamples, firstSampleIndex, n, maxPower*threshold);
    progress.end(n);
    SampleFilter::showStatistics(newSamples->count(), count());
    return newSamples;
  } else {
    return nullptr;
  }
}

void Samples::selectPower(Samples& samples, int beginIndex, int endIndex, double minPower) const
{
  for(int i=beginIndex; i<endIndex; i++) {
    const Sample& s=at(i);
    if(s.power()>minPower) {
      samples.append(s);
    }
  }
}

Samples * Samples::filterCurve(const ModalCurve& curve, double relativeRange) const
{
  TRACE;
  ConsoleProgress progress;
  progress.setCaption(tr("Filtering out around curve '%1' (>%2 %)")
                      .arg(curve.name())
                      .arg(relativeRange*100.0));
  Samples * newSamples=new Samples(*this);
  int n=count();
  double lastF=std::numeric_limits<double>::quiet_NaN();
  bool fInside=true;
  double smin=0.0, smax=0.0, ds=1.0+relativeRange;
  for(int i=0; i<n; i++) {
    const Sample& s=at(i);
    if(lastF!=s.frequency()) {
      lastF=s.frequency();
      fInside=curve.isInsideRange(s.frequency());
      const RealStatisticalPoint& p=curve.interpole(s.frequency(), LogScale, LogScale);
      smin=p.mean()/ds;
      smax=p.mean()*ds;
    }
    if(fInside && s.slowness()>smin && s.slowness()<smax) {
      newSamples->append(s);
    }
    progress.setValue(i);
  }
  progress.end(n);
  SampleFilter::showStatistics(newSamples->count(), count());
  return newSamples;
}

VectorList<Histogram2D::Sample> Samples::dispersionSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    s.append(Histogram2D::Sample(it->frequency(), it->slowness()));
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::ellipticitySamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    s.append(Histogram2D::Sample(it->frequency(), it->ellipticity()));
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::verticalNoiseSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  double v;
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    v=it->verticalNoise();
    if(isnormal(v)) {
      s.append(Histogram2D::Sample(it->frequency(), v));
    }
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::horizontalNoiseSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  double v;
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    v=it->horizontalNoise();
    if(isnormal(v)) {
      s.append(Histogram2D::Sample(it->frequency(), v));
    }
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::totalNoiseSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  double v;
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    v=it->totalNoise();
    if(isnormal(v)) {
      s.append(Histogram2D::Sample(it->frequency(), v));
    }
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::deltaNoiseSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  double v;
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    v=it->deltaNoise();
    if(isnormal(v)) {
      s.append(Histogram2D::Sample(it->frequency(), v));
    }
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::sigmaNoiseSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  double v;
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    v=it->sigmaNoise();
    if(isnormal(v)) {
      s.append(Histogram2D::Sample(it->frequency(), v));
    }
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::azimuthSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    s.append(Histogram2D::Sample(it->frequency(), it->azimuth()));
  }
  return s;
}

VectorList<Histogram2D::Sample> Samples::powerSamples() const
{
  VectorList<Histogram2D::Sample> s;
  s.reserve(count());
  for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
    s.append(Histogram2D::Sample(it->frequency(), it->power()));
  }
  return s;
}

void Samples::setValid(const Histogram2D& o)
{
  ASSERT(count()==o.sampleCount());

  for(int i=count()-1; i>=0; i--) {
    Sample& s=(*this)[i];
    s.setValid(o.sample(i).isValid());
  }
}
bool Samples::save(const QString& fileName) const
{
  TRACE;
  QFile f(fileName);
  if(f.open(QIODevice::WriteOnly)) {
    QTextStream s(&f);
    s.setRealNumberPrecision(20);
    if(!_headerLine.isEmpty()) {
       s << _headerLine;
    }
    for(VectorList<Sample>::const_iterator it=begin(); it!=end(); it++) {
      if(it->isValid()) {
        it->writeLine(s);
      }
    }
    return true;
  } else {
    return false;
  }

}

